<?php
declare(strict_types=1);

use App\Controllers\AuthController;
use App\Controllers\BackupController;
use App\Controllers\BorrowedMoneyController;
use App\Controllers\CategoryController;
use App\Controllers\ExpenseController;
use App\Controllers\HomeController;
use App\Controllers\IncomeController;
use App\Controllers\MigrationController;
use App\Controllers\RecurringController;
use App\Controllers\ReminderController;
use App\Controllers\ReportController;
use App\Controllers\RoleController;
use App\Controllers\SearchController;
use App\Controllers\SentMoneyController;
use App\Controllers\SettingsController;
use App\Controllers\UserController;
use App\Core\Router;
use App\Services\AuthService;

define('BASE_PATH', dirname(__DIR__));

require BASE_PATH . '/app/bootstrap.php';

$router = new Router();

$homeController = new HomeController();
$authController = new AuthController();
$migrationController = new MigrationController();
$settingsController = new SettingsController();
$categoryController = new CategoryController();
$incomeController = new IncomeController();
$expenseController = new ExpenseController();
$sentMoneyController = new SentMoneyController();
$borrowedMoneyController = new BorrowedMoneyController();
$reminderController = new ReminderController();
$recurringController = new RecurringController();
$searchController = new SearchController();
$reportController = new ReportController();
$backupController = new BackupController();
$userController = new UserController();
$roleController = new RoleController();

$guard = static function (callable|array $handler, ?string $permission = null): callable {
    return static function () use ($handler, $permission): void {
        if (!AuthService::check()) {
            flash('error', 'يرجى تسجيل الدخول أولاً.');
            redirect('/login');
        }

        if ($permission !== null && !AuthService::can($permission)) {
            http_response_code(403);
            ob_start();
            require BASE_PATH . '/app/Views/errors/403.php';
            $output = (string) ob_get_clean();
            echo translate_text($output);
            return;
        }

        call_user_func($handler);
    };
};

$router->get('/health', [$homeController, 'health']);
$router->get('/migrate', [$migrationController, 'run']);

$router->get('/login', [$authController, 'showLogin']);
$router->post('/login', [$authController, 'login']);
$router->get('/register', [$authController, 'showRegister']);
$router->post('/register', [$authController, 'register']);
$router->post('/logout', $guard([$authController, 'logout']));

$router->get('/', $guard([$homeController, 'index'], 'dashboard.view'));

$router->get('/settings', $guard([$settingsController, 'index'], 'settings.manage'));
$router->post('/settings', $guard([$settingsController, 'update'], 'settings.manage'));

$router->get('/categories', $guard([$categoryController, 'index'], 'categories.manage'));
$router->post('/categories/store', $guard([$categoryController, 'store'], 'categories.manage'));
$router->get('/categories/edit', $guard([$categoryController, 'edit'], 'categories.manage'));
$router->post('/categories/update', $guard([$categoryController, 'update'], 'categories.manage'));
$router->post('/categories/deactivate', $guard([$categoryController, 'deactivate'], 'categories.manage'));

$router->get('/incomes', $guard([$incomeController, 'index'], 'incomes.view'));
$router->post('/incomes/store', $guard([$incomeController, 'store'], 'incomes.manage'));
$router->get('/incomes/edit', $guard([$incomeController, 'edit'], 'incomes.manage'));
$router->post('/incomes/update', $guard([$incomeController, 'update'], 'incomes.manage'));
$router->post('/incomes/delete', $guard([$incomeController, 'delete'], 'incomes.manage'));

$router->get('/expenses', $guard([$expenseController, 'index'], 'expenses.view'));
$router->post('/expenses/store', $guard([$expenseController, 'store'], 'expenses.manage'));
$router->get('/expenses/edit', $guard([$expenseController, 'edit'], 'expenses.manage'));
$router->post('/expenses/update', $guard([$expenseController, 'update'], 'expenses.manage'));
$router->post('/expenses/delete', $guard([$expenseController, 'delete'], 'expenses.manage'));

$router->get('/sent-money', $guard([$sentMoneyController, 'index'], 'sent.view'));
$router->post('/sent-money/store', $guard([$sentMoneyController, 'store'], 'sent.manage'));
$router->get('/sent-money/edit', $guard([$sentMoneyController, 'edit'], 'sent.manage'));
$router->post('/sent-money/update', $guard([$sentMoneyController, 'update'], 'sent.manage'));
$router->post('/sent-money/delete', $guard([$sentMoneyController, 'delete'], 'sent.manage'));

$router->get('/borrowed-money', $guard([$borrowedMoneyController, 'index'], 'borrowed.view'));
$router->post('/borrowed-money/store', $guard([$borrowedMoneyController, 'store'], 'borrowed.manage'));
$router->get('/borrowed-money/edit', $guard([$borrowedMoneyController, 'edit'], 'borrowed.manage'));
$router->post('/borrowed-money/update', $guard([$borrowedMoneyController, 'update'], 'borrowed.manage'));
$router->post('/borrowed-money/delete', $guard([$borrowedMoneyController, 'delete'], 'borrowed.manage'));

$router->get('/reminders', $guard([$reminderController, 'index'], 'reminders.view'));
$router->post('/reminders/store', $guard([$reminderController, 'store'], 'reminders.manage'));
$router->get('/reminders/edit', $guard([$reminderController, 'edit'], 'reminders.manage'));
$router->post('/reminders/update', $guard([$reminderController, 'update'], 'reminders.manage'));
$router->post('/reminders/delete', $guard([$reminderController, 'delete'], 'reminders.manage'));

$router->get('/recurring', $guard([$recurringController, 'index'], 'recurring.view'));
$router->get('/recurring/incomes', $guard([$recurringController, 'incomes'], 'recurring.view'));
$router->get('/recurring/expenses', $guard([$recurringController, 'expenses'], 'recurring.view'));
$router->post('/recurring/store', $guard([$recurringController, 'store'], 'recurring.manage'));
$router->get('/recurring/edit', $guard([$recurringController, 'edit'], 'recurring.manage'));
$router->get('/recurring/incomes/edit', $guard([$recurringController, 'editIncome'], 'recurring.manage'));
$router->get('/recurring/expenses/edit', $guard([$recurringController, 'editExpense'], 'recurring.manage'));
$router->post('/recurring/update', $guard([$recurringController, 'update'], 'recurring.manage'));
$router->post('/recurring/delete', $guard([$recurringController, 'delete'], 'recurring.manage'));
$router->post('/recurring/generate', $guard([$recurringController, 'generate'], 'recurring.manage'));
$router->post('/recurring/generate-single', $guard([$recurringController, 'generateSingle'], 'recurring.manage'));

$router->get('/search', $guard([$searchController, 'index'], 'search.view'));

$router->get('/reports', $guard([$reportController, 'index'], 'reports.view'));
$router->get('/reports/export', $guard([$reportController, 'exportJson'], 'reports.view'));

$router->get('/backup', $guard([$backupController, 'index'], 'backup.manage'));
$router->get('/backup/download-db', $guard([$backupController, 'downloadDatabase'], 'backup.manage'));
$router->get('/backup/export-json', $guard([$backupController, 'exportJson'], 'backup.manage'));
$router->post('/backup/restore-sqlite', $guard([$backupController, 'restoreSqlite'], 'backup.manage'));
$router->post('/backup/import-json', $guard([$backupController, 'importJson'], 'backup.manage'));
$router->post('/backup/factory-reset', $guard([$backupController, 'factoryReset'], 'backup.manage'));

$router->get('/users', $guard([$userController, 'index'], 'users.manage'));
$router->post('/users/store', $guard([$userController, 'store'], 'users.manage'));
$router->get('/users/edit', $guard([$userController, 'edit'], 'users.manage'));
$router->post('/users/update', $guard([$userController, 'update'], 'users.manage'));
$router->post('/users/toggle-active', $guard([$userController, 'toggleActive'], 'users.manage'));

$router->get('/roles', $guard([$roleController, 'index'], 'roles.manage'));
$router->post('/roles/store', $guard([$roleController, 'store'], 'roles.manage'));
$router->post('/roles/update-role', $guard([$roleController, 'updateRole'], 'roles.manage'));
$router->post('/roles/update-permissions', $guard([$roleController, 'updatePermissions'], 'roles.manage'));

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$uri = $_SERVER['REQUEST_URI'] ?? '/';
$path = parse_url($uri, PHP_URL_PATH) ?: '/';

$baseUrl = APP_BASE_URL;
if ($baseUrl !== '' && str_starts_with($path, $baseUrl)) {
    $path = substr($path, strlen($baseUrl));
    $path = $path === '' ? '/' : $path;
}

if ($path === '/index.php') {
    $path = '/';
}

$router->dispatch($method, $path);
