<?php
declare(strict_types=1);

use App\Services\CsrfService;
use App\Services\AuthService;
use App\Services\AppSettingsService;

function e(mixed $value): string
{
    return htmlspecialchars((string) $value, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function display_description(mixed $value): string
{
    $text = (string) $value;

    // Hide internal recurring generation marker from UI display.
    $text = preg_replace('/\s*\[REC:(income|expense):[0-9a-f\-]{36}:\d{4}-\d{2}\]\s*$/i', '', $text);

    return trim((string) $text);
}

function redirect(string $path): void
{
    if (preg_match('#^https?://#i', $path)) {
        header('Location: ' . $path);
        exit;
    }

    header('Location: ' . url($path));
    exit;
}

function flash(string $key, ?string $message = null): ?string
{
    if ($message !== null) {
        $_SESSION['_flash'][$key] = $message;
        return null;
    }

    if (!isset($_SESSION['_flash'][$key])) {
        return null;
    }

    $value = (string) $_SESSION['_flash'][$key];
    unset($_SESSION['_flash'][$key]);

    return $value;
}

function set_old_input(array $input): void
{
    unset($input['_token']);
    $_SESSION['_old_input'] = $input;
}

function get_old_input(): array
{
    $old = $_SESSION['_old_input'] ?? [];
    unset($_SESSION['_old_input']);

    return is_array($old) ? $old : [];
}

function csrf_token(): string
{
    return CsrfService::token();
}

function csrf_field(): string
{
    return '<input type="hidden" name="_token" value="' . e(csrf_token()) . '">';
}

function url(string $path = '/'): string
{
    $base = defined('APP_BASE_URL') ? (string) APP_BASE_URL : '';

    if ($path === '' || $path === '/') {
        return $base === '' ? '/' : $base . '/';
    }

    if (preg_match('#^https?://#i', $path)) {
        return $path;
    }

    return ($base === '' ? '' : $base) . '/' . ltrim($path, '/');
}

function app_settings(): array
{
    return AppSettingsService::all();
}

function app_setting(string $key, mixed $default = null): mixed
{
    return AppSettingsService::get($key, $default);
}

function app_name(): string
{
    return (string) app_setting('system_name', 'مدير الدخل والمصروفات');
}

function app_lang(): string
{
    $lang = strtolower(trim((string) app_setting('language', 'ar')));
    return in_array($lang, ['ar', 'en'], true) ? $lang : 'ar';
}

function is_rtl(): bool
{
    return (int) app_setting('rtl', 1) === 1;
}

function app_dir(): string
{
    return is_rtl() ? 'rtl' : 'ltr';
}

function bootstrap_css_url(): string
{
    $file = is_rtl() ? 'bootstrap.rtl.min.css' : 'bootstrap.min.css';
    return 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/' . $file;
}

function currency_code(): string
{
    return (string) app_setting('currency_code', 'SAR');
}

function format_number(mixed $value, int $decimals = 2): string
{
    $number = (float) $value;
    $format = (string) app_setting('number_format', '1,234.56');

    if ($format === '1.234,56') {
        return number_format($number, $decimals, ',', '.');
    }

    return number_format($number, $decimals, '.', ',');
}

function format_money(mixed $value, bool $withCurrency = true): string
{
    $formatted = format_number($value, 2);
    if (!$withCurrency) {
        return $formatted;
    }

    $currency = trim(currency_code());
    return $currency === '' ? $formatted : ($formatted . ' ' . $currency);
}

function translation_map(?string $lang = null): array
{
    static $cache = [];

    $resolvedLang = $lang ?? app_lang();
    if (!in_array($resolvedLang, ['ar', 'en'], true)) {
        $resolvedLang = 'ar';
    }

    if (isset($cache[$resolvedLang])) {
        return $cache[$resolvedLang];
    }

    if ($resolvedLang === 'ar') {
        $cache[$resolvedLang] = [];
        return $cache[$resolvedLang];
    }

    $path = BASE_PATH . '/app/Lang/' . $resolvedLang . '.php';
    if (!is_file($path)) {
        $cache[$resolvedLang] = [];
        return $cache[$resolvedLang];
    }

    $map = require $path;
    $cache[$resolvedLang] = is_array($map) ? $map : [];

    return $cache[$resolvedLang];
}

function translate_text(string $text): string
{
    if (app_lang() === 'ar') {
        return $text;
    }

    $map = translation_map();
    if ($map === []) {
        return $text;
    }

    return strtr($text, $map);
}

function t(string $text): string
{
    return translate_text($text);
}

function auth_check(): bool
{
    return AuthService::check();
}

function auth_user(): ?array
{
    return AuthService::user();
}

function can(string $permission): bool
{
    return AuthService::can($permission);
}
