<?php
declare(strict_types=1);

use App\Database\Connection;
use App\Database\MigrationRunner;
use App\Models\BorrowedMoneyModel;
use App\Models\CategoryModel;
use App\Models\ExpenseModel;
use App\Models\IncomeModel;
use App\Models\RecurringModel;
use App\Models\ReminderModel;
use App\Models\SentMoneyModel;
use App\Services\BackupService;
use App\Services\ReportService;
use App\Services\SearchService;
use App\Services\UuidService;

define('BASE_PATH', dirname(__DIR__, 2));
require BASE_PATH . '/app/bootstrap.php';

function assert_true(bool $condition, string $message): void
{
    if (!$condition) {
        throw new RuntimeException('SMOKE ASSERTION FAILED: ' . $message);
    }
}

$runner = new MigrationRunner();
$runner->run();

$pdo = Connection::getInstance();
$tables = $pdo->query("SELECT name FROM sqlite_master WHERE type='table'")->fetchAll(PDO::FETCH_COLUMN);
$requiredTables = [
    'incomes',
    'expenses',
    'recurring_incomes',
    'recurring_expenses',
    'sent_money',
    'borrowed_money',
    'monthly_reminders',
    'categories',
    'app_settings',
    'recurring_generation_log',
];

foreach ($requiredTables as $table) {
    assert_true(in_array($table, $tables, true), 'Missing table: ' . $table);
}

$categoryModel = new CategoryModel();
$incomeCategory = $categoryModel->listByType('income')[0] ?? null;
$expenseCategory = $categoryModel->listByType('expense')[0] ?? null;
assert_true($incomeCategory !== null, 'No income category available');
assert_true($expenseCategory !== null, 'No expense category available');

$incomeModel = new IncomeModel();
$expenseModel = new ExpenseModel();
$sentModel = new SentMoneyModel();
$borrowedModel = new BorrowedMoneyModel();
$reminderModel = new ReminderModel();
$recurringModel = new RecurringModel();

$now = date('Y-m-d H:i:s');
$today = date('Y-m-d');
$month = date('Y-m');

$incomeId = UuidService::v4();
assert_true($incomeModel->create([
    'id' => $incomeId,
    'amount' => 100.50,
    'category_id' => $incomeCategory['id'],
    'description' => 'SMOKE_INCOME_KEYWORD',
    'date' => $today,
    'created_at' => $now,
    'updated_at' => $now,
]), 'Income create failed');
assert_true($incomeModel->findById($incomeId) !== null, 'Income read failed');

$expenseId = UuidService::v4();
assert_true($expenseModel->create([
    'id' => $expenseId,
    'amount' => 40.25,
    'category_id' => $expenseCategory['id'],
    'description' => 'SMOKE_EXPENSE_KEYWORD',
    'date' => $today,
    'created_at' => $now,
    'updated_at' => $now,
]), 'Expense create failed');
assert_true($expenseModel->findById($expenseId) !== null, 'Expense read failed');

$sentId = UuidService::v4();
assert_true($sentModel->create([
    'id' => $sentId,
    'person_name' => 'SMOKE_PERSON',
    'amount' => 20.00,
    'expected_receive_date' => $today,
    'status' => 'sent',
    'notes' => 'SMOKE_SENT',
    'created_at' => $now,
    'updated_at' => $now,
]), 'Sent create failed');

$borrowedId = UuidService::v4();
assert_true($borrowedModel->create([
    'id' => $borrowedId,
    'person_name' => 'SMOKE_PERSON',
    'amount' => 30.00,
    'expected_pay_date' => $today,
    'status' => 'borrowed',
    'notes' => 'SMOKE_BORROWED',
    'created_at' => $now,
    'updated_at' => $now,
]), 'Borrowed create failed');

$reminderId = UuidService::v4();
assert_true($reminderModel->create([
    'id' => $reminderId,
    'title' => 'SMOKE_REMINDER',
    'amount' => 10.00,
    'due_date' => $today,
    'status' => 'pending',
    'notes' => 'SMOKE_NOTES',
    'created_at' => $now,
    'updated_at' => $now,
]), 'Reminder create failed');

$recurringId = UuidService::v4();
assert_true($recurringModel->create('income', [
    'id' => $recurringId,
    'amount' => 77.77,
    'category_id' => $incomeCategory['id'],
    'description' => 'SMOKE_RECURRING',
    'day_of_month' => 1,
    'is_active' => 1,
    'created_at' => $now,
    'updated_at' => $now,
]), 'Recurring create failed');

$generation = $recurringModel->generateCurrentMonth();
assert_true(isset($generation['created'], $generation['skipped']), 'Recurring generation result invalid');

$searchService = new SearchService();
$searchResult = $searchService->search('SMOKE_', '', '');
assert_true(count($searchResult['incomes']) > 0, 'Search incomes failed');

$reportService = new ReportService();
$report = $reportService->monthlySummary($month);
assert_true(isset($report['totals']['income']), 'Report summary invalid');

$backupService = new BackupService();
$backup = $backupService->exportJsonData();
assert_true(isset($backup['tables']['incomes']), 'Backup export invalid');

$incomeModel->delete($incomeId);
$expenseModel->delete($expenseId);
$sentModel->delete($sentId);
$borrowedModel->delete($borrowedId);
$reminderModel->delete($reminderId);
$recurringModel->delete('income', $recurringId);

$pdo->prepare('DELETE FROM incomes WHERE description LIKE :desc')->execute([':desc' => 'SMOKE_RECURRING%']);
$pdo->prepare(
    'DELETE FROM recurring_generation_log
     WHERE recurring_id = :recurring_id AND recurring_type = :recurring_type'
)->execute([
    ':recurring_id' => $recurringId,
    ':recurring_type' => 'income',
]);

echo "SMOKE TEST PASSED\n";
