<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use PDO;
use App\Services\AuthService;

class SearchService
{
    private PDO $db;

    public function __construct()
    {
        $this->db = Connection::getInstance();
    }

    public function search(string $keyword, string $dateFrom, string $dateTo): array
    {
        return [
            'incomes' => $this->searchIncomes($keyword, $dateFrom, $dateTo),
            'expenses' => $this->searchExpenses($keyword, $dateFrom, $dateTo),
            'sent_money' => $this->searchSentMoney($keyword, $dateFrom, $dateTo),
            'borrowed_money' => $this->searchBorrowedMoney($keyword, $dateFrom, $dateTo),
        ];
    }

    private function searchIncomes(string $keyword, string $dateFrom, string $dateTo): array
    {
        $params = [];
        $conditions = $this->keywordCondition('i.description', $keyword, $params, 'kw');
        $conditions = array_merge($conditions, $this->dateCondition('i.date', $dateFrom, $dateTo, $params, 'income'));
        $this->appendUserScope($conditions, $params, 'i.user_id', ':scope_user_id');

        $sql = 'SELECT i.id, i.amount, i.description, i.date,
                       c.name AS category_name, c.is_active AS category_is_active
                FROM incomes i
                LEFT JOIN categories c ON c.id = i.category_id';
        if ($conditions !== []) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }
        $sql .= ' ORDER BY i.date DESC, i.created_at DESC LIMIT 200';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    private function searchExpenses(string $keyword, string $dateFrom, string $dateTo): array
    {
        $params = [];
        $conditions = $this->keywordCondition('e.description', $keyword, $params, 'kw');
        $conditions = array_merge($conditions, $this->dateCondition('e.date', $dateFrom, $dateTo, $params, 'expense'));
        $this->appendUserScope($conditions, $params, 'e.user_id', ':scope_user_id');

        $sql = 'SELECT e.id, e.amount, e.description, e.date,
                       c.name AS category_name, c.is_active AS category_is_active
                FROM expenses e
                LEFT JOIN categories c ON c.id = e.category_id';
        if ($conditions !== []) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }
        $sql .= ' ORDER BY e.date DESC, e.created_at DESC LIMIT 200';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    private function searchSentMoney(string $keyword, string $dateFrom, string $dateTo): array
    {
        $params = [];
        $conditions = [];

        if ($keyword !== '') {
            $params[':kw'] = '%' . $keyword . '%';
            $conditions[] = '(person_name LIKE :kw OR notes LIKE :kw)';
        }

        $conditions = array_merge($conditions, $this->dateCondition('expected_receive_date', $dateFrom, $dateTo, $params, 'sent'));
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $sql = 'SELECT id, person_name, amount, expected_receive_date, status, notes
                FROM sent_money';
        if ($conditions !== []) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }
        $sql .= ' ORDER BY expected_receive_date DESC, created_at DESC LIMIT 200';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    private function searchBorrowedMoney(string $keyword, string $dateFrom, string $dateTo): array
    {
        $params = [];
        $conditions = [];

        if ($keyword !== '') {
            $params[':kw'] = '%' . $keyword . '%';
            $conditions[] = '(person_name LIKE :kw OR notes LIKE :kw)';
        }

        $conditions = array_merge($conditions, $this->dateCondition('expected_pay_date', $dateFrom, $dateTo, $params, 'borrowed'));
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $sql = 'SELECT id, person_name, amount, expected_pay_date, status, notes
                FROM borrowed_money';
        if ($conditions !== []) {
            $sql .= ' WHERE ' . implode(' AND ', $conditions);
        }
        $sql .= ' ORDER BY expected_pay_date DESC, created_at DESC LIMIT 200';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    private function keywordCondition(string $column, string $keyword, array &$params, string $prefix): array
    {
        if ($keyword === '') {
            return [];
        }

        $key = ':' . $prefix;
        $params[$key] = '%' . $keyword . '%';
        return [$column . ' LIKE ' . $key];
    }

    private function dateCondition(string $column, string $dateFrom, string $dateTo, array &$params, string $prefix): array
    {
        $conditions = [];

        if ($dateFrom !== '') {
            $key = ':from_' . $prefix;
            $conditions[] = $column . ' >= ' . $key;
            $params[$key] = $dateFrom;
        }

        if ($dateTo !== '') {
            $key = ':to_' . $prefix;
            $conditions[] = $column . ' <= ' . $key;
            $params[$key] = $dateTo;
        }

        return $conditions;
    }

    private function appendUserScope(array &$conditions, array &$params, string $column, string $paramKey): void
    {
        if (!AuthService::check() || AuthService::isAdmin()) {
            return;
        }

        $userId = AuthService::id();
        if ($userId === null || $userId === '') {
            $conditions[] = '1 = 0';
            return;
        }

        $conditions[] = $column . ' = ' . $paramKey;
        $params[$paramKey] = $userId;
    }
}
