<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use PDO;
use App\Services\AuthService;

class ReportService
{
    private PDO $db;

    public function __construct()
    {
        $this->db = Connection::getInstance();
    }

    public function monthlySummary(string $month): array
    {
        [$startDate, $endDate] = $this->monthRange($month);

        $totals = [
            'income' => $this->sumByRange('incomes', $startDate, $endDate),
            'expense' => $this->sumByRange('expenses', $startDate, $endDate),
        ];

        return [
            'month' => $month,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'totals' => [
                'income' => $totals['income'],
                'expense' => $totals['expense'],
                'balance' => $totals['income'] - $totals['expense'],
            ],
            'breakdown' => [
                'income' => $this->categoryBreakdown('incomes', $startDate, $endDate),
                'expense' => $this->categoryBreakdown('expenses', $startDate, $endDate),
            ],
        ];
    }

    private function sumByRange(string $table, string $startDate, string $endDate): float
    {
        $params = [
            ':start_date' => $startDate,
            ':end_date' => $endDate,
        ];
        $conditions = [
            'date >= :start_date',
            'date < :end_date',
        ];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT COALESCE(SUM(amount), 0) AS total
             FROM ' . $table . '
             WHERE ' . implode(' AND ', $conditions)
        );
        $stmt->execute($params);

        return (float) $stmt->fetch()['total'];
    }

    private function categoryBreakdown(string $table, string $startDate, string $endDate): array
    {
        $params = [
            ':start_date' => $startDate,
            ':end_date' => $endDate,
        ];
        $conditions = [
            't.date >= :start_date',
            't.date < :end_date',
        ];
        $this->appendUserScope($conditions, $params, 't.user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT
                COALESCE(c.name, "غير مصنف") AS category_name,
                c.is_active AS category_is_active,
                COALESCE(SUM(t.amount), 0) AS total_amount
             FROM ' . $table . ' t
             LEFT JOIN categories c ON c.id = t.category_id
             WHERE ' . implode(' AND ', $conditions) . '
             GROUP BY category_name, c.is_active
             ORDER BY total_amount DESC'
        );
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    private function monthRange(string $month): array
    {
        $start = \DateTime::createFromFormat('Y-m', $month);
        if (!$start) {
            throw new \InvalidArgumentException('Invalid month format. Expected YYYY-MM.');
        }

        $start->setDate((int) $start->format('Y'), (int) $start->format('m'), 1);
        $end = clone $start;
        $end->modify('+1 month');

        return [$start->format('Y-m-d'), $end->format('Y-m-d')];
    }

    private function appendUserScope(array &$conditions, array &$params, string $column, string $paramKey): void
    {
        if (!AuthService::check() || AuthService::isAdmin()) {
            return;
        }

        $userId = AuthService::id();
        if ($userId === null || $userId === '') {
            $conditions[] = '1 = 0';
            return;
        }

        $conditions[] = $column . ' = ' . $paramKey;
        $params[$paramKey] = $userId;
    }
}
