<?php
declare(strict_types=1);

namespace App\Services;

use ErrorException;
use Throwable;

class ErrorHandler
{
    private static string $logFile;

    public static function register(string $logFile): void
    {
        self::$logFile = $logFile;

        set_error_handler(static function (
            int $severity,
            string $message,
            string $file,
            int $line
        ): bool {
            if (!(error_reporting() & $severity)) {
                return false;
            }

            throw new ErrorException($message, 0, $severity, $file, $line);
        });

        set_exception_handler([self::class, 'handleException']);

        register_shutdown_function(static function (): void {
            $error = error_get_last();

            if ($error === null) {
                return;
            }

            $fatalTypes = [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_PARSE];

            if (!in_array($error['type'], $fatalTypes, true)) {
                return;
            }

            self::handleException(new ErrorException(
                $error['message'],
                0,
                $error['type'],
                $error['file'],
                $error['line']
            ));
        });
    }

    public static function handleException(Throwable $exception): void
    {
        $errorId = bin2hex(random_bytes(6));

        self::writeLog($errorId, $exception);

        if (!headers_sent()) {
            http_response_code(500);
        }

        $errorView = BASE_PATH . '/app/Views/errors/500.php';

        if (is_file($errorView)) {
            ob_start();
            require $errorView;
            $output = (string) ob_get_clean();
            echo translate_text($output);
            return;
        }

        echo t('حدث خطأ غير متوقع. رقم الخطأ: ') . $errorId;
    }

    private static function writeLog(string $errorId, Throwable $exception): void
    {
        $line = sprintf(
            "[%s] ERROR_ID=%s %s in %s:%d\n%s\n\n",
            date('Y-m-d H:i:s'),
            $errorId,
            $exception->getMessage(),
            $exception->getFile(),
            $exception->getLine(),
            $exception->getTraceAsString()
        );

        file_put_contents(self::$logFile, $line, FILE_APPEND);
    }
}
