<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use PDO;
use App\Services\AuthService;

class DashboardService
{
    private PDO $db;

    public function __construct()
    {
        $this->db = Connection::getInstance();
    }

    public function stats(array $filters = []): array
    {
        $totalIncome = $this->sum('incomes', 'date', $filters);
        $totalExpense = $this->sum('expenses', 'date', $filters);
        $totalSent = $this->sum('sent_money', 'expected_receive_date', $filters);
        $totalBorrowed = $this->sum('borrowed_money', 'expected_pay_date', $filters);
        $currentBalance = $totalIncome - $totalExpense;

        $activeSent = $this->sumWithStatuses(
            'sent_money',
            ['sent', 'overdue'],
            'expected_receive_date',
            $filters
        );
        $activeBorrowed = $this->sumWithStatuses(
            'borrowed_money',
            ['borrowed', 'overdue', 'rescheduled'],
            'expected_pay_date',
            $filters
        );
        $effectiveBalance = $currentBalance - $activeSent + $activeBorrowed;

        $today = date('Y-m-d');
        [$remindersScopeSql, $remindersScopeParams] = $this->scopeClause('monthly_reminders', ':scope_alert_user_id');
        [$sentScopeSql, $sentScopeParams] = $this->scopeClause('sent_money', ':scope_alert_user_id');
        [$borrowedScopeSql, $borrowedScopeParams] = $this->scopeClause('borrowed_money', ':scope_alert_user_id');

        $alerts = [
            'overdue_reminders' => $this->count(
                'SELECT COUNT(*) FROM monthly_reminders
                 WHERE status = :status AND due_date < :today' . $remindersScopeSql,
                array_merge([':status' => 'pending', ':today' => $today], $remindersScopeParams)
            ),
            'overdue_sent' => $this->count(
                'SELECT COUNT(*) FROM sent_money
                 WHERE (status = :overdue OR (status = :sent AND expected_receive_date < :today))' . $sentScopeSql,
                array_merge([':overdue' => 'overdue', ':sent' => 'sent', ':today' => $today], $sentScopeParams)
            ),
            'overdue_borrowed' => $this->count(
                'SELECT COUNT(*) FROM borrowed_money
                 WHERE (status = :overdue OR (status = :borrowed AND expected_pay_date < :today))' . $borrowedScopeSql,
                array_merge([':overdue' => 'overdue', ':borrowed' => 'borrowed', ':today' => $today], $borrowedScopeParams)
            ),
        ];

        return [
            'total_income' => $totalIncome,
            'total_expense' => $totalExpense,
            'total_sent' => $totalSent,
            'total_borrowed' => $totalBorrowed,
            'current_balance' => $currentBalance,
            'effective_balance' => $effectiveBalance,
            'active_sent' => $activeSent,
            'active_borrowed' => $activeBorrowed,
            'alerts' => $alerts,
        ];
    }

    private function sum(string $table, ?string $dateColumn = null, array $filters = []): float
    {
        $params = [];
        $conditions = [];
        $this->appendDateRangeConditions($conditions, $params, $dateColumn, $filters, 'sum');
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : (' WHERE ' . implode(' AND ', $conditions));
        $sql = 'SELECT COALESCE(SUM(amount), 0) AS total FROM ' . $table . $where;
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (float) $stmt->fetchColumn();
    }

    private function sumWithStatuses(string $table, array $statuses, ?string $dateColumn = null, array $filters = []): float
    {
        $placeholders = [];
        $params = [];

        foreach ($statuses as $index => $status) {
            $key = ':status_' . $index;
            $placeholders[] = $key;
            $params[$key] = $status;
        }

        $sql = 'SELECT COALESCE(SUM(amount), 0) AS total FROM ' . $table
            . ' WHERE status IN (' . implode(',', $placeholders) . ')';
        $conditions = [];
        $this->appendDateRangeConditions($conditions, $params, $dateColumn, $filters, 'status');
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        if ($conditions !== []) {
            $sql .= ' AND ' . implode(' AND ', $conditions);
        }

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return (float) $stmt->fetchColumn();
    }

    private function count(string $sql, array $params): int
    {
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (int) $stmt->fetchColumn();
    }

    private function appendDateRangeConditions(array &$conditions, array &$params, ?string $dateColumn, array $filters, string $prefix): void
    {
        if ($dateColumn === null) {
            return;
        }

        $dateFrom = trim((string) ($filters['date_from'] ?? ''));
        $dateTo = trim((string) ($filters['date_to'] ?? ''));

        if ($dateFrom !== '') {
            $dateFromKey = ':' . $prefix . '_date_from';
            $conditions[] = $dateColumn . ' >= ' . $dateFromKey;
            $params[$dateFromKey] = $dateFrom;
        }

        if ($dateTo !== '') {
            $dateToKey = ':' . $prefix . '_date_to';
            $conditions[] = $dateColumn . ' <= ' . $dateToKey;
            $params[$dateToKey] = $dateTo;
        }
    }

    private function appendUserScope(array &$conditions, array &$params, string $column, string $key): void
    {
        if (!$this->shouldApplyUserScope()) {
            return;
        }

        $userId = AuthService::id();
        if ($userId === null || $userId === '') {
            $conditions[] = '1 = 0';
            return;
        }

        $conditions[] = $column . ' = ' . $key;
        $params[$key] = $userId;
    }

    private function shouldApplyUserScope(): bool
    {
        return AuthService::check() && !AuthService::isAdmin();
    }

    private function scopeClause(string $table, string $paramKey): array
    {
        if (!$this->shouldApplyUserScope()) {
            return ['', []];
        }

        $userId = AuthService::id();
        if ($userId === null || $userId === '') {
            return [' AND 1 = 0', []];
        }

        return [' AND ' . $table . '.user_id = ' . $paramKey, [$paramKey => $userId]];
    }
}
