<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use App\Database\MigrationRunner;
use PDO;
use RuntimeException;

class BackupService
{
    private PDO $db;
    private string $databasePath;
    private string $backupsPath;

    public function __construct()
    {
        $this->db = Connection::getInstance();
        $this->databasePath = Connection::databasePath();
        $this->backupsPath = STORAGE_PATH . '/backups';
    }

    public function databasePath(): string
    {
        return $this->databasePath;
    }

    public function exportJsonData(): array
    {
        $tables = [
            'app_settings',
            'categories',
            'incomes',
            'expenses',
            'recurring_incomes',
            'recurring_expenses',
            'sent_money',
            'borrowed_money',
            'monthly_reminders',
            'recurring_generation_log',
        ];

        $data = [];
        foreach ($tables as $table) {
            if (!$this->tableExists($table)) {
                $data[$table] = [];
                continue;
            }

            $stmt = $this->db->query('SELECT * FROM ' . $table);
            $data[$table] = $stmt->fetchAll() ?: [];
        }

        return [
            'exported_at' => date('Y-m-d H:i:s'),
            'tables' => $data,
        ];
    }

    public function restoreDatabaseFromFile(string $sourcePath): void
    {
        if (!is_file($sourcePath)) {
            throw new RuntimeException('ملف قاعدة البيانات المرفوع غير صالح.');
        }

        $this->backupCurrentDatabase('pre-restore-sqlite');

        Connection::reset();

        if (!copy($sourcePath, $this->databasePath)) {
            throw new RuntimeException('تعذر استبدال ملف قاعدة البيانات.');
        }

        Connection::reset();
        $pdo = Connection::getInstance();
        $pdo->query('SELECT 1');
        $this->db = $pdo;
    }

    public function importJsonReplace(string $jsonContent): void
    {
        $decoded = json_decode($jsonContent, true);
        if (!is_array($decoded) || !isset($decoded['tables']) || !is_array($decoded['tables'])) {
            throw new RuntimeException('ملف JSON غير صالح.');
        }

        $tables = $decoded['tables'];
        $order = [
            'app_settings',
            'categories',
            'incomes',
            'expenses',
            'recurring_incomes',
            'recurring_expenses',
            'sent_money',
            'borrowed_money',
            'monthly_reminders',
            'recurring_generation_log',
        ];

        $this->backupCurrentDatabase('pre-import-json');

        $this->db->beginTransaction();
        try {
            foreach (array_reverse($order) as $table) {
                if ($this->tableExists($table)) {
                    $this->db->exec('DELETE FROM ' . $table);
                }
            }

            foreach ($order as $table) {
                if (!$this->tableExists($table)) {
                    continue;
                }

                $rows = $tables[$table] ?? [];
                if (!is_array($rows) || $rows === []) {
                    continue;
                }

                foreach ($rows as $row) {
                    if (!is_array($row) || $row === []) {
                        continue;
                    }

                    $columns = array_keys($row);
                    $placeholders = array_map(static fn ($column) => ':' . $column, $columns);
                    $sql = 'INSERT INTO ' . $table . ' (' . implode(',', $columns) . ') VALUES (' . implode(',', $placeholders) . ')';
                    $stmt = $this->db->prepare($sql);
                    $params = [];
                    foreach ($columns as $column) {
                        $params[':' . $column] = $row[$column];
                    }
                    $stmt->execute($params);
                }
            }

            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }

    public function factoryReset(): void
    {
        $this->backupCurrentDatabase('pre-factory-reset');

        Connection::reset();

        if (is_file($this->databasePath) && !unlink($this->databasePath)) {
            throw new RuntimeException('تعذر حذف قاعدة البيانات الحالية.');
        }

        touch($this->databasePath);
        Connection::reset();

        $runner = new MigrationRunner();
        $runner->run();
        $this->db = Connection::getInstance();
    }

    private function backupCurrentDatabase(string $prefix): void
    {
        if (!is_file($this->databasePath)) {
            return;
        }

        $fileName = $prefix . '-' . date('Ymd-His') . '.sqlite';
        $destination = $this->backupsPath . '/' . $fileName;
        copy($this->databasePath, $destination);
    }

    private function tableExists(string $table): bool
    {
        $stmt = $this->db->prepare(
            'SELECT COUNT(*) FROM sqlite_master WHERE type = :type AND name = :name'
        );
        $stmt->execute([
            ':type' => 'table',
            ':name' => $table,
        ]);

        return (int) $stmt->fetchColumn() > 0;
    }
}
