<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use PDO;
use Throwable;

class AuthService
{
    private const SESSION_KEY = '_auth_user_id';

    private static ?array $cachedUser = null;
    private static ?array $cachedPermissions = null;
    private static ?array $cachedRoles = null;

    public static function login(string $email, string $password): bool
    {
        try {
            $db = Connection::getInstance();
            $stmt = $db->prepare(
                'SELECT id, name, email, password_hash, is_active
                 FROM users
                 WHERE email = :email
                 LIMIT 1'
            );
            $stmt->execute([':email' => $email]);
            $user = $stmt->fetch();
        } catch (Throwable) {
            return false;
        }

        if (!$user || (int) $user['is_active'] !== 1) {
            return false;
        }

        if (!password_verify($password, (string) $user['password_hash'])) {
            return false;
        }

        $_SESSION[self::SESSION_KEY] = (string) $user['id'];
        if (!headers_sent()) {
            session_regenerate_id(true);
        }

        self::$cachedUser = null;
        self::$cachedPermissions = null;
        self::$cachedRoles = null;

        return true;
    }

    public static function logout(): void
    {
        unset($_SESSION[self::SESSION_KEY]);
        self::$cachedUser = null;
        self::$cachedPermissions = null;
        self::$cachedRoles = null;
        if (!headers_sent()) {
            session_regenerate_id(true);
        }
    }

    public static function check(): bool
    {
        return self::user() !== null;
    }

    public static function user(): ?array
    {
        if (self::$cachedUser !== null) {
            return self::$cachedUser;
        }

        $userId = self::id();
        if ($userId === null) {
            return null;
        }

        try {
            $db = Connection::getInstance();
            $stmt = $db->prepare(
                'SELECT id, name, email, is_active, created_at, updated_at
                 FROM users
                 WHERE id = :id
                 LIMIT 1'
            );
            $stmt->execute([':id' => $userId]);
            $user = $stmt->fetch();
        } catch (Throwable) {
            self::logout();
            return null;
        }

        if (!$user || (int) $user['is_active'] !== 1) {
            self::logout();
            return null;
        }

        self::$cachedUser = $user;
        return self::$cachedUser;
    }

    public static function id(): ?string
    {
        $id = $_SESSION[self::SESSION_KEY] ?? null;
        return is_string($id) && $id !== '' ? $id : null;
    }

    public static function can(string $permissionName): bool
    {
        if (!self::check()) {
            return false;
        }

        return in_array($permissionName, self::permissions(), true);
    }

    public static function roles(): array
    {
        if (self::$cachedRoles !== null) {
            return self::$cachedRoles;
        }

        $userId = self::id();
        if ($userId === null) {
            return [];
        }

        try {
            $db = Connection::getInstance();
            $stmt = $db->prepare(
                'SELECT DISTINCT r.name
                 FROM user_roles ur
                 INNER JOIN roles r ON r.id = ur.role_id AND r.is_active = 1
                 WHERE ur.user_id = :user_id'
            );
            $stmt->execute([':user_id' => $userId]);
            $rows = $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];
        } catch (Throwable) {
            return [];
        }

        self::$cachedRoles = array_values(array_unique(array_map('strval', $rows)));
        return self::$cachedRoles;
    }

    public static function hasRole(string $roleName): bool
    {
        if (!self::check()) {
            return false;
        }

        return in_array($roleName, self::roles(), true);
    }

    public static function isAdmin(): bool
    {
        return self::hasRole('admin');
    }

    public static function permissions(): array
    {
        if (self::$cachedPermissions !== null) {
            return self::$cachedPermissions;
        }

        $userId = self::id();
        if ($userId === null) {
            return [];
        }

        try {
            $db = Connection::getInstance();
            $stmt = $db->prepare(
                'SELECT DISTINCT p.name
                 FROM user_roles ur
                 INNER JOIN roles r ON r.id = ur.role_id AND r.is_active = 1
                 INNER JOIN role_permissions rp ON rp.role_id = r.id
                 INNER JOIN permissions p ON p.id = rp.permission_id
                 WHERE ur.user_id = :user_id'
            );
            $stmt->execute([':user_id' => $userId]);
            $rows = $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];
        } catch (Throwable) {
            return [];
        }

        self::$cachedPermissions = array_values(array_unique(array_map('strval', $rows)));
        return self::$cachedPermissions;
    }
}
