<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use Throwable;

class AppSettingsService
{
    private static ?array $cached = null;

    public static function all(): array
    {
        if (self::$cached !== null) {
            return self::$cached;
        }

        $defaults = [
            'system_name' => 'مدير الدخل والمصروفات',
            'currency_code' => 'SAR',
            'number_format' => '1,234.56',
            'calendar_type' => 'gregorian',
            'language' => 'ar',
            'rtl' => 1,
        ];

        try {
            $db = Connection::getInstance();
            $stmt = $db->query(
                'SELECT system_name, currency_code, number_format, calendar_type, language, rtl
                 FROM app_settings
                 ORDER BY created_at ASC
                 LIMIT 1'
            );
            $row = $stmt->fetch();
            $settings = is_array($row) ? array_merge($defaults, $row) : $defaults;
        } catch (Throwable) {
            $settings = $defaults;
        }

        $systemName = trim((string) ($settings['system_name'] ?? ''));
        $settings['system_name'] = $systemName !== '' ? $systemName : $defaults['system_name'];

        $currencyCode = strtoupper(trim((string) ($settings['currency_code'] ?? '')));
        $settings['currency_code'] = $currencyCode !== '' ? $currencyCode : $defaults['currency_code'];

        $numberFormat = (string) ($settings['number_format'] ?? $defaults['number_format']);
        if (!in_array($numberFormat, ['1,234.56', '1.234,56'], true)) {
            $numberFormat = $defaults['number_format'];
        }
        $settings['number_format'] = $numberFormat;

        $calendarType = (string) ($settings['calendar_type'] ?? $defaults['calendar_type']);
        if (!in_array($calendarType, ['gregorian', 'hijri'], true)) {
            $calendarType = $defaults['calendar_type'];
        }
        $settings['calendar_type'] = $calendarType;

        $language = trim((string) ($settings['language'] ?? $defaults['language']));
        $settings['language'] = $language !== '' ? $language : $defaults['language'];

        $settings['rtl'] = ((int) ($settings['rtl'] ?? 1) === 1) ? 1 : 0;

        self::$cached = $settings;
        return self::$cached;
    }

    public static function get(string $key, mixed $default = null): mixed
    {
        $settings = self::all();
        return $settings[$key] ?? $default;
    }
}
