<?php
declare(strict_types=1);

namespace App\Models;

class SentMoneyModel extends BaseModel
{
    public function list(string $statusFilter): array
    {
        $params = [':today' => date('Y-m-d')];
        $conditions = [];

        if ($statusFilter !== '' && $statusFilter !== 'all') {
            $conditions[] = 's.status = :status';
            $params[':status'] = $statusFilter;
        }
        $this->appendUserScope($conditions, $params, 's.user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : ('WHERE ' . implode(' AND ', $conditions));

        $sql = 'SELECT s.*,
                       CASE
                           WHEN s.status = "overdue" THEN 1
                           WHEN s.status = "sent" AND s.expected_receive_date < :today THEN 1
                           ELSE 0
                       END AS is_overdue
                FROM sent_money s
                ' . $where . '
                ORDER BY s.expected_receive_date ASC, s.created_at DESC';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    public function findById(string $id): ?array
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT *
             FROM sent_money
             WHERE ' . implode(' AND ', $conditions) . '
             LIMIT 1'
        );
        $stmt->execute($params);
        $row = $stmt->fetch();

        return $row ?: null;
    }

    public function create(array $data): bool
    {
        $userId = $this->resolveInsertUserId($data);
        $stmt = $this->db->prepare(
            'INSERT INTO sent_money
            (id, user_id, person_name, amount, expected_receive_date, status, notes, created_at, updated_at)
            VALUES (:id, :user_id, :person_name, :amount, :expected_receive_date, :status, :notes, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':user_id' => $userId,
            ':person_name' => $data['person_name'],
            ':amount' => $data['amount'],
            ':expected_receive_date' => $data['expected_receive_date'],
            ':status' => $data['status'],
            ':notes' => $data['notes'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function update(array $data): bool
    {
        $params = [
            ':id' => $data['id'],
            ':person_name' => $data['person_name'],
            ':amount' => $data['amount'],
            ':expected_receive_date' => $data['expected_receive_date'],
            ':status' => $data['status'],
            ':notes' => $data['notes'],
            ':updated_at' => $data['updated_at'],
        ];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'UPDATE sent_money
             SET person_name = :person_name,
                 amount = :amount,
                 expected_receive_date = :expected_receive_date,
                 status = :status,
                 notes = :notes,
                 updated_at = :updated_at
             WHERE ' . implode(' AND ', $conditions)
        );

        return $stmt->execute($params);
    }

    public function delete(string $id): bool
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        $stmt = $this->db->prepare('DELETE FROM sent_money WHERE ' . implode(' AND ', $conditions));
        return $stmt->execute($params);
    }
}
