<?php
declare(strict_types=1);

namespace App\Models;

class ReminderModel extends BaseModel
{
    public function list(string $statusFilter): array
    {
        $params = [':today' => date('Y-m-d')];
        $conditions = [];

        if ($statusFilter !== '' && $statusFilter !== 'all') {
            $conditions[] = 'r.status = :status';
            $params[':status'] = $statusFilter;
        }
        $this->appendUserScope($conditions, $params, 'r.user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : ('WHERE ' . implode(' AND ', $conditions));

        $sql = 'SELECT r.*,
                       CASE
                           WHEN r.status = "pending" AND r.due_date < :today THEN 1
                           ELSE 0
                       END AS is_overdue
                FROM monthly_reminders r
                ' . $where . '
                ORDER BY r.due_date ASC, r.created_at DESC';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    public function findById(string $id): ?array
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT *
             FROM monthly_reminders
             WHERE ' . implode(' AND ', $conditions) . '
             LIMIT 1'
        );
        $stmt->execute($params);
        $row = $stmt->fetch();

        return $row ?: null;
    }

    public function create(array $data): bool
    {
        $userId = $this->resolveInsertUserId($data);
        $stmt = $this->db->prepare(
            'INSERT INTO monthly_reminders
            (id, user_id, title, amount, due_date, status, notes, created_at, updated_at)
            VALUES (:id, :user_id, :title, :amount, :due_date, :status, :notes, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':user_id' => $userId,
            ':title' => $data['title'],
            ':amount' => $data['amount'],
            ':due_date' => $data['due_date'],
            ':status' => $data['status'],
            ':notes' => $data['notes'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function update(array $data): bool
    {
        $params = [
            ':id' => $data['id'],
            ':title' => $data['title'],
            ':amount' => $data['amount'],
            ':due_date' => $data['due_date'],
            ':status' => $data['status'],
            ':notes' => $data['notes'],
            ':updated_at' => $data['updated_at'],
        ];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'UPDATE monthly_reminders
             SET title = :title,
                 amount = :amount,
                 due_date = :due_date,
                 status = :status,
                 notes = :notes,
                 updated_at = :updated_at
             WHERE ' . implode(' AND ', $conditions)
        );

        return $stmt->execute($params);
    }

    public function delete(string $id): bool
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        $stmt = $this->db->prepare('DELETE FROM monthly_reminders WHERE ' . implode(' AND ', $conditions));
        return $stmt->execute($params);
    }
}
