<?php
declare(strict_types=1);

namespace App\Models;

class RecurringModel extends BaseModel
{
    public function list(string $type): array
    {
        $table = $this->tableFromType($type);
        $params = [];
        $conditions = [];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : (' WHERE ' . implode(' AND ', $conditions));

        $stmt = $this->db->prepare(
            'SELECT * FROM ' . $table . $where . ' ORDER BY is_active DESC, day_of_month ASC, created_at DESC'
        );
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    public function findById(string $type, string $id): ?array
    {
        $table = $this->tableFromType($type);
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT * FROM ' . $table . ' WHERE ' . implode(' AND ', $conditions) . ' LIMIT 1'
        );
        $stmt->execute($params);
        $row = $stmt->fetch();

        return $row ?: null;
    }

    public function create(string $type, array $data): bool
    {
        $table = $this->tableFromType($type);
        $userId = $this->resolveInsertUserId($data);
        $stmt = $this->db->prepare(
            'INSERT INTO ' . $table . '
            (id, user_id, amount, category_id, description, day_of_month, is_active, created_at, updated_at)
            VALUES (:id, :user_id, :amount, :category_id, :description, :day_of_month, :is_active, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':user_id' => $userId,
            ':amount' => $data['amount'],
            ':category_id' => $data['category_id'],
            ':description' => $data['description'],
            ':day_of_month' => $data['day_of_month'],
            ':is_active' => $data['is_active'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function update(string $type, array $data): bool
    {
        $table = $this->tableFromType($type);
        $params = [
            ':id' => $data['id'],
            ':amount' => $data['amount'],
            ':category_id' => $data['category_id'],
            ':description' => $data['description'],
            ':day_of_month' => $data['day_of_month'],
            ':is_active' => $data['is_active'],
            ':updated_at' => $data['updated_at'],
        ];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'UPDATE ' . $table . '
             SET amount = :amount,
                 category_id = :category_id,
                 description = :description,
                 day_of_month = :day_of_month,
                 is_active = :is_active,
                 updated_at = :updated_at
             WHERE ' . implode(' AND ', $conditions)
        );

        return $stmt->execute($params);
    }

    public function delete(string $type, string $id): bool
    {
        $table = $this->tableFromType($type);
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare('DELETE FROM ' . $table . ' WHERE ' . implode(' AND ', $conditions));
        return $stmt->execute($params);
    }

    public function generateCurrentMonth(): array
    {
        $year = (int) date('Y');
        $month = (int) date('m');
        $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);
        $now = date('Y-m-d H:i:s');

        $created = 0;
        $skipped = 0;

        $this->db->beginTransaction();
        try {
            $created += $this->generateForType('income', 'recurring_incomes', 'incomes', $year, $month, $daysInMonth, $now, $skipped);
            $created += $this->generateForType('expense', 'recurring_expenses', 'expenses', $year, $month, $daysInMonth, $now, $skipped);
            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }

        return [
            'created' => $created,
            'skipped' => $skipped,
            'year' => $year,
            'month' => $month,
        ];
    }

    public function generateCurrentMonthForType(string $type): array
    {
        $resolvedType = $type === 'expense' ? 'expense' : 'income';
        $year = (int) date('Y');
        $month = (int) date('m');
        $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);
        $now = date('Y-m-d H:i:s');

        $created = 0;
        $skipped = 0;

        $this->db->beginTransaction();
        try {
            $created += $this->generateForType(
                $resolvedType,
                $this->tableFromType($resolvedType),
                $this->targetTableFromType($resolvedType),
                $year,
                $month,
                $daysInMonth,
                $now,
                $skipped
            );
            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }

        return [
            'created' => $created,
            'skipped' => $skipped,
            'year' => $year,
            'month' => $month,
        ];
    }

    public function generateSingleCurrentMonth(string $type, string $recurringId): array
    {
        $resolvedType = $type === 'expense' ? 'expense' : 'income';
        $year = (int) date('Y');
        $month = (int) date('m');
        $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);
        $now = date('Y-m-d H:i:s');

        $created = 0;
        $skipped = 0;

        $this->db->beginTransaction();
        try {
            $created += $this->generateForType(
                $resolvedType,
                $this->tableFromType($resolvedType),
                $this->targetTableFromType($resolvedType),
                $year,
                $month,
                $daysInMonth,
                $now,
                $skipped,
                $recurringId
            );
            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }

        return [
            'created' => $created,
            'skipped' => $skipped,
            'year' => $year,
            'month' => $month,
        ];
    }

    private function generateForType(
        string $type,
        string $sourceTable,
        string $targetTable,
        int $year,
        int $month,
        int $daysInMonth,
        string $now,
        int &$skipped,
        ?string $onlyRecurringId = null
    ): int {
        $created = 0;

        $rowsSql = 'SELECT id, user_id, amount, category_id, description, day_of_month
             FROM ' . $sourceTable;
        $rowsParams = [];
        $rowsConditions = ['is_active = 1'];
        if ($onlyRecurringId !== null) {
            $rowsConditions[] = 'id = :id';
            $rowsParams[':id'] = $onlyRecurringId;
        }
        $this->appendUserScope($rowsConditions, $rowsParams, 'user_id', ':scope_user_id');
        $rowsSql .= ' WHERE ' . implode(' AND ', $rowsConditions);

        $rowsStmt = $this->db->prepare($rowsSql);
        $rowsStmt->execute($rowsParams);
        $rows = $rowsStmt->fetchAll() ?: [];

        $checkStmt = $this->db->prepare(
            'SELECT COUNT(*) FROM recurring_generation_log
             WHERE recurring_type = :recurring_type
               AND recurring_id = :recurring_id
               AND year = :year
               AND month = :month'
        );
        $deleteLogStmt = $this->db->prepare(
            'DELETE FROM recurring_generation_log
             WHERE recurring_type = :recurring_type
               AND recurring_id = :recurring_id
               AND year = :year
               AND month = :month'
        );
        $existsGeneratedStmt = $this->db->prepare(
            'SELECT COUNT(*) FROM ' . $targetTable . '
             WHERE date = :date
               AND amount = :amount
               AND ((user_id = :user_id) OR (user_id IS NULL AND :user_id IS NULL))
               AND ((category_id = :category_id) OR (category_id IS NULL AND :category_id IS NULL))
               AND (description = :description_legacy OR description = :description_with_marker)'
        );

        $insertTarget = $this->db->prepare(
            'INSERT INTO ' . $targetTable . '
             (id, user_id, amount, category_id, description, date, created_at, updated_at)
             VALUES (:id, :user_id, :amount, :category_id, :description, :date, :created_at, :updated_at)'
        );

        $insertLog = $this->db->prepare(
            'INSERT INTO recurring_generation_log
             (id, recurring_type, recurring_id, year, month, created_at)
             VALUES (:id, :recurring_type, :recurring_id, :year, :month, :created_at)'
        );

        foreach ($rows as $row) {
            $day = max(1, min((int) $row['day_of_month'], $daysInMonth));
            $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
            $monthLabel = sprintf('%04d-%02d', $year, $month);
            $legacyDescription = $row['description'] . ' (متكرر ' . $monthLabel . ')';
            $descriptionWithMarker = $legacyDescription . ' ' . $this->generationMarker($type, (string) $row['id'], $year, $month);
            $categoryId = $row['category_id'] === '' ? null : $row['category_id'];
            $amount = (float) $row['amount'];
            $userId = $row['user_id'] === '' ? null : $row['user_id'];

            $checkStmt->execute([
                ':recurring_type' => $type,
                ':recurring_id' => $row['id'],
                ':year' => $year,
                ':month' => $month,
            ]);

            if ((int) $checkStmt->fetchColumn() > 0) {
                if ($this->generatedTransactionExists(
                    $existsGeneratedStmt,
                    $date,
                    $amount,
                    $userId,
                    $categoryId,
                    $legacyDescription,
                    $descriptionWithMarker
                )) {
                    $skipped++;
                    continue;
                }

                // Log exists but generated row does not exist anymore (deleted manually): treat as stale.
                $deleteLogStmt->execute([
                    ':recurring_type' => $type,
                    ':recurring_id' => $row['id'],
                    ':year' => $year,
                    ':month' => $month,
                ]);
            }

            $insertTarget->execute([
                ':id' => \App\Services\UuidService::v4(),
                ':user_id' => $userId,
                ':amount' => $amount,
                ':category_id' => $categoryId,
                ':description' => $descriptionWithMarker,
                ':date' => $date,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);

            $insertLog->execute([
                ':id' => \App\Services\UuidService::v4(),
                ':recurring_type' => $type,
                ':recurring_id' => $row['id'],
                ':year' => $year,
                ':month' => $month,
                ':created_at' => $now,
            ]);

            $created++;
        }

        return $created;
    }

    private function tableFromType(string $type): string
    {
        return $type === 'expense' ? 'recurring_expenses' : 'recurring_incomes';
    }

    private function targetTableFromType(string $type): string
    {
        return $type === 'expense' ? 'expenses' : 'incomes';
    }

    private function generationMarker(string $type, string $recurringId, int $year, int $month): string
    {
        return '[REC:' . $type . ':' . $recurringId . ':' . sprintf('%04d-%02d', $year, $month) . ']';
    }

    private function generatedTransactionExists(
        \PDOStatement $stmt,
        string $date,
        float $amount,
        ?string $userId,
        ?string $categoryId,
        string $legacyDescription,
        string $descriptionWithMarker
    ): bool {
        $stmt->execute([
            ':date' => $date,
            ':amount' => $amount,
            ':user_id' => $userId,
            ':category_id' => $categoryId,
            ':description_legacy' => $legacyDescription,
            ':description_with_marker' => $descriptionWithMarker,
        ]);

        return (int) $stmt->fetchColumn() > 0;
    }
}
