<?php
declare(strict_types=1);

namespace App\Models;

class IncomeModel extends BaseModel
{
    public function list(array $filters): array
    {
        $params = [];
        $conditions = $this->buildDateFilter($filters, $params, 'i.date');
        $this->appendUserScope($conditions, $params, 'i.user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : ('WHERE ' . implode(' AND ', $conditions));

        $sql = 'SELECT i.id, i.amount, i.category_id, i.description, i.date, i.created_at, i.updated_at,
                       c.name AS category_name, c.is_active AS category_is_active
                FROM incomes i
                LEFT JOIN categories c ON c.id = i.category_id
                ' . $where . '
                ORDER BY i.date DESC, i.created_at DESC';

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll() ?: [];
    }

    public function summary(array $filters): array
    {
        $params = [];
        $conditions = $this->buildDateFilter($filters, $params, 'date');
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');
        $where = $conditions === [] ? '' : ('WHERE ' . implode(' AND ', $conditions));

        $sql = 'SELECT COUNT(*) AS total_count, COALESCE(SUM(amount), 0) AS total_amount
                FROM incomes
                ' . $where;

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $row = $stmt->fetch();

        return [
            'count' => (int) ($row['total_count'] ?? 0),
            'amount' => (float) ($row['total_amount'] ?? 0),
        ];
    }

    public function findById(string $id): ?array
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'SELECT id, amount, category_id, description, date, created_at, updated_at, user_id
             FROM incomes
             WHERE ' . implode(' AND ', $conditions) . '
             LIMIT 1'
        );
        $stmt->execute($params);
        $row = $stmt->fetch();

        return $row ?: null;
    }

    public function create(array $data): bool
    {
        $userId = $this->resolveInsertUserId($data);
        $stmt = $this->db->prepare(
            'INSERT INTO incomes (id, user_id, amount, category_id, description, date, created_at, updated_at)
             VALUES (:id, :user_id, :amount, :category_id, :description, :date, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':user_id' => $userId,
            ':amount' => $data['amount'],
            ':category_id' => $data['category_id'],
            ':description' => $data['description'],
            ':date' => $data['date'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function update(array $data): bool
    {
        $params = [
            ':id' => $data['id'],
            ':amount' => $data['amount'],
            ':category_id' => $data['category_id'],
            ':description' => $data['description'],
            ':date' => $data['date'],
            ':updated_at' => $data['updated_at'],
        ];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare(
            'UPDATE incomes
             SET amount = :amount,
                 category_id = :category_id,
                 description = :description,
                 date = :date,
                 updated_at = :updated_at
             WHERE ' . implode(' AND ', $conditions)
        );

        return $stmt->execute($params);
    }

    public function delete(string $id): bool
    {
        $params = [':id' => $id];
        $conditions = ['id = :id'];
        $this->appendUserScope($conditions, $params, 'user_id', ':scope_user_id');

        $stmt = $this->db->prepare('DELETE FROM incomes WHERE ' . implode(' AND ', $conditions));
        return $stmt->execute($params);
    }

    private function buildDateFilter(array $filters, array &$params, string $column): array
    {
        $conditions = [];

        if (($filters['date_from'] ?? '') !== '') {
            $conditions[] = $column . ' >= :date_from';
            $params[':date_from'] = $filters['date_from'];
        }

        if (($filters['date_to'] ?? '') !== '') {
            $conditions[] = $column . ' <= :date_to';
            $params[':date_to'] = $filters['date_to'];
        }

        return $conditions;
    }
}
