<?php
declare(strict_types=1);

namespace App\Models;

class AclModel extends BaseModel
{
    public function listUsers(): array
    {
        $stmt = $this->db->query(
            'SELECT u.id, u.name, u.email, u.is_active, u.created_at, u.updated_at,
                    COALESCE(GROUP_CONCAT(r.display_name, ", "), "") AS roles_text
             FROM users u
             LEFT JOIN user_roles ur ON ur.user_id = u.id
             LEFT JOIN roles r ON r.id = ur.role_id
             GROUP BY u.id, u.name, u.email, u.is_active, u.created_at, u.updated_at
             ORDER BY u.created_at DESC'
        );

        return $stmt->fetchAll() ?: [];
    }

    public function findUserById(string $id): ?array
    {
        $stmt = $this->db->prepare(
            'SELECT id, name, email, is_active, created_at, updated_at
             FROM users
             WHERE id = :id
             LIMIT 1'
        );
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch();

        return $row ?: null;
    }

    public function createUser(array $data): bool
    {
        $stmt = $this->db->prepare(
            'INSERT INTO users (id, name, email, password_hash, is_active, created_at, updated_at)
             VALUES (:id, :name, :email, :password_hash, :is_active, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':name' => $data['name'],
            ':email' => $data['email'],
            ':password_hash' => $data['password_hash'],
            ':is_active' => $data['is_active'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function updateUser(array $data, bool $updatePassword): bool
    {
        if ($updatePassword) {
            $stmt = $this->db->prepare(
                'UPDATE users
                 SET name = :name,
                     email = :email,
                     password_hash = :password_hash,
                     is_active = :is_active,
                     updated_at = :updated_at
                 WHERE id = :id'
            );

            return $stmt->execute([
                ':id' => $data['id'],
                ':name' => $data['name'],
                ':email' => $data['email'],
                ':password_hash' => $data['password_hash'],
                ':is_active' => $data['is_active'],
                ':updated_at' => $data['updated_at'],
            ]);
        }

        $stmt = $this->db->prepare(
            'UPDATE users
             SET name = :name,
                 email = :email,
                 is_active = :is_active,
                 updated_at = :updated_at
             WHERE id = :id'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':name' => $data['name'],
            ':email' => $data['email'],
            ':is_active' => $data['is_active'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function setUserActive(string $id, int $isActive): bool
    {
        $stmt = $this->db->prepare(
            'UPDATE users
             SET is_active = :is_active,
                 updated_at = :updated_at
             WHERE id = :id'
        );

        return $stmt->execute([
            ':id' => $id,
            ':is_active' => $isActive,
            ':updated_at' => date('Y-m-d H:i:s'),
        ]);
    }

    public function emailExists(string $email, ?string $excludeId = null): bool
    {
        $sql = 'SELECT COUNT(*) FROM users WHERE email = :email';
        $params = [':email' => $email];

        if ($excludeId !== null) {
            $sql .= ' AND id != :exclude_id';
            $params[':exclude_id'] = $excludeId;
        }

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (int) $stmt->fetchColumn() > 0;
    }

    public function listRoles(bool $onlyActive = false): array
    {
        $sql = 'SELECT id, name, display_name, is_active, created_at, updated_at
                FROM roles';
        if ($onlyActive) {
            $sql .= ' WHERE is_active = 1';
        }
        $sql .= ' ORDER BY display_name ASC';

        $stmt = $this->db->query($sql);
        return $stmt->fetchAll() ?: [];
    }

    public function findRoleById(string $id): ?array
    {
        $stmt = $this->db->prepare(
            'SELECT id, name, display_name, is_active, created_at, updated_at
             FROM roles WHERE id = :id LIMIT 1'
        );
        $stmt->execute([':id' => $id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    public function findRoleIdByName(string $name, bool $onlyActive = true): ?string
    {
        $sql = 'SELECT id FROM roles WHERE name = :name';
        if ($onlyActive) {
            $sql .= ' AND is_active = 1';
        }
        $sql .= ' LIMIT 1';

        $stmt = $this->db->prepare($sql);
        $stmt->execute([':name' => $name]);
        $id = $stmt->fetchColumn();

        return is_string($id) && $id !== '' ? $id : null;
    }

    public function roleNameExists(string $name, ?string $excludeId = null): bool
    {
        $sql = 'SELECT COUNT(*) FROM roles WHERE name = :name';
        $params = [':name' => $name];

        if ($excludeId !== null) {
            $sql .= ' AND id != :exclude_id';
            $params[':exclude_id'] = $excludeId;
        }

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (int) $stmt->fetchColumn() > 0;
    }

    public function createRole(array $data): bool
    {
        $stmt = $this->db->prepare(
            'INSERT INTO roles (id, name, display_name, is_active, created_at, updated_at)
             VALUES (:id, :name, :display_name, :is_active, :created_at, :updated_at)'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':name' => $data['name'],
            ':display_name' => $data['display_name'],
            ':is_active' => $data['is_active'],
            ':created_at' => $data['created_at'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function updateRole(array $data): bool
    {
        $stmt = $this->db->prepare(
            'UPDATE roles
             SET display_name = :display_name,
                 is_active = :is_active,
                 updated_at = :updated_at
             WHERE id = :id'
        );

        return $stmt->execute([
            ':id' => $data['id'],
            ':display_name' => $data['display_name'],
            ':is_active' => $data['is_active'],
            ':updated_at' => $data['updated_at'],
        ]);
    }

    public function listPermissions(): array
    {
        $stmt = $this->db->query(
            'SELECT id, name, display_name, created_at, updated_at
             FROM permissions
             ORDER BY display_name ASC'
        );

        return $stmt->fetchAll() ?: [];
    }

    public function listPermissionIds(): array
    {
        $stmt = $this->db->query('SELECT id FROM permissions');
        $rows = $stmt->fetchAll() ?: [];
        $ids = [];
        foreach ($rows as $row) {
            $ids[] = (string) $row['id'];
        }

        return $ids;
    }

    public function getUserRoleIds(string $userId): array
    {
        $stmt = $this->db->prepare('SELECT role_id FROM user_roles WHERE user_id = :user_id');
        $stmt->execute([':user_id' => $userId]);
        $rows = $stmt->fetchAll() ?: [];
        $result = [];
        foreach ($rows as $row) {
            $result[] = (string) $row['role_id'];
        }
        return $result;
    }

    public function setUserRoles(string $userId, array $roleIds): void
    {
        $roleIds = array_values(array_unique(array_map('strval', $roleIds)));

        $this->db->beginTransaction();
        try {
            $deleteStmt = $this->db->prepare('DELETE FROM user_roles WHERE user_id = :user_id');
            $deleteStmt->execute([':user_id' => $userId]);

            if ($roleIds !== []) {
                $insertStmt = $this->db->prepare(
                    'INSERT INTO user_roles (user_id, role_id, created_at)
                     VALUES (:user_id, :role_id, :created_at)'
                );
                $now = date('Y-m-d H:i:s');
                foreach ($roleIds as $roleId) {
                    $insertStmt->execute([
                        ':user_id' => $userId,
                        ':role_id' => $roleId,
                        ':created_at' => $now,
                    ]);
                }
            }

            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }

    public function getRolePermissionIds(string $roleId): array
    {
        $stmt = $this->db->prepare('SELECT permission_id FROM role_permissions WHERE role_id = :role_id');
        $stmt->execute([':role_id' => $roleId]);
        $rows = $stmt->fetchAll() ?: [];
        $result = [];
        foreach ($rows as $row) {
            $result[] = (string) $row['permission_id'];
        }

        return $result;
    }

    public function setRolePermissions(string $roleId, array $permissionIds): void
    {
        $permissionIds = array_values(array_unique(array_map('strval', $permissionIds)));

        $this->db->beginTransaction();
        try {
            $deleteStmt = $this->db->prepare('DELETE FROM role_permissions WHERE role_id = :role_id');
            $deleteStmt->execute([':role_id' => $roleId]);

            if ($permissionIds !== []) {
                $insertStmt = $this->db->prepare(
                    'INSERT INTO role_permissions (role_id, permission_id, created_at)
                     VALUES (:role_id, :permission_id, :created_at)'
                );
                $now = date('Y-m-d H:i:s');
                foreach ($permissionIds as $permissionId) {
                    $insertStmt->execute([
                        ':role_id' => $roleId,
                        ':permission_id' => $permissionId,
                        ':created_at' => $now,
                    ]);
                }
            }

            $this->db->commit();
        } catch (\Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}
