<?php
declare(strict_types=1);

use App\Services\UuidService;

return [
    'name' => '20260220_000004_enforce_admin_user_and_data_ownership',
    'up' => static function (\PDO $db): void {
        $db->exec('PRAGMA foreign_keys = ON');
        $now = date('Y-m-d H:i:s');

        $hasColumn = static function (string $table, string $column) use ($db): bool {
            $stmt = $db->query('PRAGMA table_info(' . $table . ')');
            $rows = $stmt ? ($stmt->fetchAll() ?: []) : [];
            foreach ($rows as $row) {
                if (isset($row['name']) && (string) $row['name'] === $column) {
                    return true;
                }
            }
            return false;
        };

        $ownedTables = [
            'incomes',
            'expenses',
            'recurring_incomes',
            'recurring_expenses',
            'sent_money',
            'borrowed_money',
            'monthly_reminders',
        ];

        foreach ($ownedTables as $table) {
            if (!$hasColumn($table, 'user_id')) {
                $db->exec('ALTER TABLE ' . $table . ' ADD COLUMN user_id TEXT NULL');
            }
            $db->exec('CREATE INDEX IF NOT EXISTS idx_' . $table . '_user_id ON ' . $table . '(user_id)');
        }

        $adminRoleIdStmt = $db->prepare('SELECT id FROM roles WHERE name = :name LIMIT 1');
        $adminRoleIdStmt->execute([':name' => 'admin']);
        $adminRoleId = (string) ($adminRoleIdStmt->fetchColumn() ?: '');

        if ($adminRoleId === '') {
            $adminRoleId = UuidService::v4();
            $insertAdminRole = $db->prepare(
                'INSERT INTO roles (id, name, display_name, is_active, created_at, updated_at)
                 VALUES (:id, :name, :display_name, :is_active, :created_at, :updated_at)'
            );
            $insertAdminRole->execute([
                ':id' => $adminRoleId,
                ':name' => 'admin',
                ':display_name' => 'مدير النظام',
                ':is_active' => 1,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);
        }

        $userRoleIdStmt = $db->prepare('SELECT id FROM roles WHERE name = :name LIMIT 1');
        $userRoleIdStmt->execute([':name' => 'user']);
        $userRoleId = (string) ($userRoleIdStmt->fetchColumn() ?: '');

        if ($userRoleId === '') {
            $userRoleId = UuidService::v4();
            $insertUserRole = $db->prepare(
                'INSERT INTO roles (id, name, display_name, is_active, created_at, updated_at)
                 VALUES (:id, :name, :display_name, :is_active, :created_at, :updated_at)'
            );
            $insertUserRole->execute([
                ':id' => $userRoleId,
                ':name' => 'user',
                ':display_name' => 'مستخدم',
                ':is_active' => 1,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);
        }

        $db->exec("UPDATE roles SET is_active = 0, updated_at = '" . $now . "' WHERE name NOT IN ('admin', 'user')");
        $db->exec("UPDATE roles SET is_active = 1, updated_at = '" . $now . "' WHERE name IN ('admin', 'user')");

        $permissionsRows = $db->query('SELECT id, name FROM permissions')->fetchAll() ?: [];
        $permissionByName = [];
        foreach ($permissionsRows as $row) {
            $permissionByName[(string) $row['name']] = (string) $row['id'];
        }

        $adminPermissions = array_keys($permissionByName);
        $userPermissions = [
            'dashboard.view',
            'incomes.view',
            'incomes.manage',
            'expenses.view',
            'expenses.manage',
            'recurring.view',
            'recurring.manage',
            'sent.view',
            'sent.manage',
            'borrowed.view',
            'borrowed.manage',
            'reminders.view',
            'reminders.manage',
            'reports.view',
            'search.view',
        ];

        $deleteRolePermissions = $db->prepare('DELETE FROM role_permissions WHERE role_id = :role_id');
        $insertRolePermission = $db->prepare(
            'INSERT OR IGNORE INTO role_permissions (role_id, permission_id, created_at)
             VALUES (:role_id, :permission_id, :created_at)'
        );

        $applyPermissions = static function (string $roleId, array $permissionNames) use ($deleteRolePermissions, $insertRolePermission, $permissionByName, $now): void {
            $deleteRolePermissions->execute([':role_id' => $roleId]);
            foreach ($permissionNames as $permissionName) {
                if (!isset($permissionByName[$permissionName])) {
                    continue;
                }
                $insertRolePermission->execute([
                    ':role_id' => $roleId,
                    ':permission_id' => $permissionByName[$permissionName],
                    ':created_at' => $now,
                ]);
            }
        };

        $applyPermissions($adminRoleId, $adminPermissions);
        $applyPermissions($userRoleId, $userPermissions);

        // Keep only admin/user role assignments.
        $db->exec("DELETE FROM user_roles WHERE role_id NOT IN ('" . $adminRoleId . "', '" . $userRoleId . "')");

        // Ensure all non-admin users have user role.
        $insertUserRoleLink = $db->prepare(
            'INSERT OR IGNORE INTO user_roles (user_id, role_id, created_at)
             VALUES (:user_id, :role_id, :created_at)'
        );
        $nonAdminUsers = $db->query(
            'SELECT u.id
             FROM users u
             WHERE NOT EXISTS (
                 SELECT 1
                 FROM user_roles ur
                 INNER JOIN roles r ON r.id = ur.role_id
                 WHERE ur.user_id = u.id AND r.name = "admin"
             )'
        )->fetchAll() ?: [];

        foreach ($nonAdminUsers as $userRow) {
            $insertUserRoleLink->execute([
                ':user_id' => (string) $userRow['id'],
                ':role_id' => $userRoleId,
                ':created_at' => $now,
            ]);
        }

        $adminUserIdStmt = $db->query(
            'SELECT u.id
             FROM users u
             INNER JOIN user_roles ur ON ur.user_id = u.id
             INNER JOIN roles r ON r.id = ur.role_id
             WHERE r.name = "admin"
             ORDER BY u.created_at ASC
             LIMIT 1'
        );
        $adminUserId = (string) ($adminUserIdStmt ? ($adminUserIdStmt->fetchColumn() ?: '') : '');

        if ($adminUserId === '') {
            $fallbackUserIdStmt = $db->query('SELECT id FROM users ORDER BY created_at ASC LIMIT 1');
            $adminUserId = (string) ($fallbackUserIdStmt ? ($fallbackUserIdStmt->fetchColumn() ?: '') : '');
        }

        if ($adminUserId !== '') {
            foreach ($ownedTables as $table) {
                $update = $db->prepare(
                    'UPDATE ' . $table . '
                     SET user_id = :user_id
                     WHERE user_id IS NULL OR TRIM(user_id) = ""'
                );
                $update->execute([':user_id' => $adminUserId]);
            }
        }
    },
];

