<?php
declare(strict_types=1);

use App\Services\UuidService;

return [
    'name' => '20260217_000003_create_user_access_control',
    'up' => static function (\PDO $db): void {
        $db->exec('PRAGMA foreign_keys = ON');

        $db->exec(
            'CREATE TABLE IF NOT EXISTS users (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL,
                email TEXT NOT NULL UNIQUE,
                password_hash TEXT NOT NULL,
                is_active INTEGER NOT NULL DEFAULT 1,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )'
        );

        $db->exec(
            'CREATE TABLE IF NOT EXISTS roles (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL UNIQUE,
                display_name TEXT NOT NULL,
                is_active INTEGER NOT NULL DEFAULT 1,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )'
        );

        $db->exec(
            'CREATE TABLE IF NOT EXISTS permissions (
                id TEXT PRIMARY KEY,
                name TEXT NOT NULL UNIQUE,
                display_name TEXT NOT NULL,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )'
        );

        $db->exec(
            'CREATE TABLE IF NOT EXISTS user_roles (
                user_id TEXT NOT NULL,
                role_id TEXT NOT NULL,
                created_at TEXT NOT NULL,
                PRIMARY KEY (user_id, role_id),
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE
            )'
        );

        $db->exec(
            'CREATE TABLE IF NOT EXISTS role_permissions (
                role_id TEXT NOT NULL,
                permission_id TEXT NOT NULL,
                created_at TEXT NOT NULL,
                PRIMARY KEY (role_id, permission_id),
                FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
                FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE
            )'
        );

        $db->exec('CREATE INDEX IF NOT EXISTS idx_users_email ON users(email)');
        $db->exec('CREATE INDEX IF NOT EXISTS idx_user_roles_user ON user_roles(user_id)');
        $db->exec('CREATE INDEX IF NOT EXISTS idx_user_roles_role ON user_roles(role_id)');
        $db->exec('CREATE INDEX IF NOT EXISTS idx_role_permissions_role ON role_permissions(role_id)');
        $db->exec('CREATE INDEX IF NOT EXISTS idx_role_permissions_permission ON role_permissions(permission_id)');

        $now = date('Y-m-d H:i:s');

        $permissionDefinitions = [
            ['dashboard.view', 'عرض لوحة التحكم'],
            ['settings.manage', 'إدارة الإعدادات'],
            ['categories.manage', 'إدارة التصنيفات'],
            ['incomes.view', 'عرض الإيرادات'],
            ['incomes.manage', 'إدارة الإيرادات'],
            ['expenses.view', 'عرض المصروفات'],
            ['expenses.manage', 'إدارة المصروفات'],
            ['recurring.view', 'عرض المعاملات المتكررة'],
            ['recurring.manage', 'إدارة المعاملات المتكررة'],
            ['sent.view', 'عرض المبالغ المرسلة'],
            ['sent.manage', 'إدارة المبالغ المرسلة'],
            ['borrowed.view', 'عرض المبالغ المقترضة'],
            ['borrowed.manage', 'إدارة المبالغ المقترضة'],
            ['reminders.view', 'عرض التذكيرات'],
            ['reminders.manage', 'إدارة التذكيرات'],
            ['reports.view', 'عرض التقارير'],
            ['search.view', 'استخدام البحث'],
            ['backup.manage', 'إدارة النسخ الاحتياطي'],
            ['users.manage', 'إدارة المستخدمين'],
            ['roles.manage', 'إدارة الصلاحيات'],
        ];

        $insertPermission = $db->prepare(
            'INSERT OR IGNORE INTO permissions (id, name, display_name, created_at, updated_at)
             VALUES (:id, :name, :display_name, :created_at, :updated_at)'
        );

        foreach ($permissionDefinitions as [$name, $displayName]) {
            $insertPermission->execute([
                ':id' => UuidService::v4(),
                ':name' => $name,
                ':display_name' => $displayName,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);
        }

        $roleDefinitions = [
            ['admin', 'مدير النظام'],
            ['manager', 'مدير'],
            ['viewer', 'مشاهد'],
        ];

        $insertRole = $db->prepare(
            'INSERT OR IGNORE INTO roles (id, name, display_name, is_active, created_at, updated_at)
             VALUES (:id, :name, :display_name, :is_active, :created_at, :updated_at)'
        );

        foreach ($roleDefinitions as [$name, $displayName]) {
            $insertRole->execute([
                ':id' => UuidService::v4(),
                ':name' => $name,
                ':display_name' => $displayName,
                ':is_active' => 1,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);
        }

        $rolesByName = [];
        $roleRows = $db->query('SELECT id, name FROM roles')->fetchAll() ?: [];
        foreach ($roleRows as $row) {
            $rolesByName[(string) $row['name']] = (string) $row['id'];
        }

        $permissionsByName = [];
        $permissionRows = $db->query('SELECT id, name FROM permissions')->fetchAll() ?: [];
        foreach ($permissionRows as $row) {
            $permissionsByName[(string) $row['name']] = (string) $row['id'];
        }

        $managerPermissions = [
            'dashboard.view',
            'settings.manage',
            'categories.manage',
            'incomes.view',
            'incomes.manage',
            'expenses.view',
            'expenses.manage',
            'recurring.view',
            'recurring.manage',
            'sent.view',
            'sent.manage',
            'borrowed.view',
            'borrowed.manage',
            'reminders.view',
            'reminders.manage',
            'reports.view',
            'search.view',
        ];

        $viewerPermissions = [
            'dashboard.view',
            'incomes.view',
            'expenses.view',
            'recurring.view',
            'sent.view',
            'borrowed.view',
            'reminders.view',
            'reports.view',
            'search.view',
        ];

        $rolePermissionMap = [
            'admin' => array_keys($permissionsByName),
            'manager' => $managerPermissions,
            'viewer' => $viewerPermissions,
        ];

        $insertRolePermission = $db->prepare(
            'INSERT OR IGNORE INTO role_permissions (role_id, permission_id, created_at)
             VALUES (:role_id, :permission_id, :created_at)'
        );

        foreach ($rolePermissionMap as $roleName => $permissionNames) {
            if (!isset($rolesByName[$roleName])) {
                continue;
            }

            foreach ($permissionNames as $permissionName) {
                if (!isset($permissionsByName[$permissionName])) {
                    continue;
                }

                $insertRolePermission->execute([
                    ':role_id' => $rolesByName[$roleName],
                    ':permission_id' => $permissionsByName[$permissionName],
                    ':created_at' => $now,
                ]);
            }
        }

        $adminEmail = 'admin@admin.com';
        $adminExistsStmt = $db->prepare('SELECT COUNT(*) FROM users WHERE email = :email');
        $adminExistsStmt->execute([':email' => $adminEmail]);
        $adminExists = (int) $adminExistsStmt->fetchColumn() > 0;

        if (!$adminExists) {
            $adminId = UuidService::v4();

            $insertAdmin = $db->prepare(
                'INSERT INTO users (id, name, email, password_hash, is_active, created_at, updated_at)
                 VALUES (:id, :name, :email, :password_hash, :is_active, :created_at, :updated_at)'
            );
            $insertAdmin->execute([
                ':id' => $adminId,
                ':name' => 'مدير النظام',
                ':email' => $adminEmail,
                ':password_hash' => password_hash('Admin@123', PASSWORD_DEFAULT),
                ':is_active' => 1,
                ':created_at' => $now,
                ':updated_at' => $now,
            ]);

            if (isset($rolesByName['admin'])) {
                $insertUserRole = $db->prepare(
                    'INSERT OR IGNORE INTO user_roles (user_id, role_id, created_at)
                     VALUES (:user_id, :role_id, :created_at)'
                );
                $insertUserRole->execute([
                    ':user_id' => $adminId,
                    ':role_id' => $rolesByName['admin'],
                    ':created_at' => $now,
                ]);
            }
        }
    },
];
