<?php
declare(strict_types=1);

namespace App\Database;

use PDO;
use RuntimeException;
use Throwable;

class MigrationRunner
{
    private PDO $db;
    private string $migrationsPath;

    public function __construct(?PDO $db = null, ?string $migrationsPath = null)
    {
        $this->db = $db ?? Connection::getInstance();
        $this->migrationsPath = $migrationsPath ?? BASE_PATH . '/app/Database/migrations';
    }

    public function run(): array
    {
        $this->ensureMigrationsTable();

        $applied = $this->appliedMigrations();
        $files = glob($this->migrationsPath . '/*.php') ?: [];
        sort($files, SORT_STRING);

        $ran = [];
        $skipped = [];

        foreach ($files as $file) {
            $migration = require $file;

            if (!is_array($migration) || !isset($migration['name'], $migration['up']) || !is_callable($migration['up'])) {
                throw new RuntimeException('Invalid migration format: ' . basename($file));
            }

            $name = (string) $migration['name'];

            if (isset($applied[$name])) {
                $skipped[] = $name;
                continue;
            }

            $this->db->beginTransaction();

            try {
                $migration['up']($this->db);

                $insert = $this->db->prepare(
                    'INSERT INTO migrations (migration, applied_at) VALUES (:migration, :applied_at)'
                );
                $insert->execute([
                    ':migration' => $name,
                    ':applied_at' => date('Y-m-d H:i:s'),
                ]);

                $this->db->commit();
                $ran[] = $name;
                $applied[$name] = true;
            } catch (Throwable $exception) {
                $this->db->rollBack();
                throw $exception;
            }
        }

        return [
            'total_files' => count($files),
            'ran' => $ran,
            'skipped' => $skipped,
        ];
    }

    private function ensureMigrationsTable(): void
    {
        $this->db->exec(
            'CREATE TABLE IF NOT EXISTS migrations (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                migration TEXT NOT NULL UNIQUE,
                applied_at TEXT NOT NULL
            )'
        );
    }

    private function appliedMigrations(): array
    {
        $stmt = $this->db->query('SELECT migration FROM migrations');
        $rows = $stmt->fetchAll();
        $result = [];

        foreach ($rows as $row) {
            $result[(string) $row['migration']] = true;
        }

        return $result;
    }
}
