<?php
declare(strict_types=1);

namespace App\Core;

use App\Services\CsrfService;
use RuntimeException;

class Router
{
    private array $routes = [
        'GET' => [],
        'POST' => [],
    ];

    public function get(string $path, callable|array $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }

    public function post(string $path, callable|array $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }

    public function dispatch(string $method, string $path): void
    {
        $method = strtoupper($method);
        $normalizedPath = $this->normalizePath($path);

        if (!isset($this->routes[$method])) {
            http_response_code(405);
            echo t('طريقة الطلب غير مدعومة');
            return;
        }

        if ($method === 'POST') {
            $token = $_POST['_token'] ?? null;
            if (!CsrfService::verify(is_string($token) ? $token : null)) {
                throw new RuntimeException('Invalid CSRF token.');
            }
        }

        $handler = $this->routes[$method][$normalizedPath] ?? null;

        if ($handler === null) {
            http_response_code(404);
            echo t('الصفحة غير موجودة');
            return;
        }

        if (!is_callable($handler)) {
            throw new RuntimeException('Route handler is not callable.');
        }

        call_user_func($handler);
    }

    private function addRoute(string $method, string $path, callable|array $handler): void
    {
        $normalizedPath = $this->normalizePath($path);
        $this->routes[$method][$normalizedPath] = $handler;
    }

    private function normalizePath(string $path): string
    {
        $path = '/' . trim($path, '/');

        if ($path === '//') {
            return '/';
        }

        $trimmed = rtrim($path, '/');
        return $trimmed === '' ? '/' : $trimmed;
    }
}
