<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\AclModel;
use App\Services\AuthService;
use App\Services\UuidService;

class UserController extends Controller
{
    private AclModel $aclModel;

    public function __construct()
    {
        $this->aclModel = new AclModel();
    }

    public function index(): void
    {
        $this->render('users/index', [
            'title' => 'إدارة المستخدمين',
            'users' => $this->aclModel->listUsers(),
            'roles' => $this->aclModel->listRoles(true),
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validate($_POST, false);
        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/users');
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $userId = UuidService::v4();

        $this->aclModel->createUser([
            'id' => $userId,
            'name' => $payload['name'],
            'email' => $payload['email'],
            'password_hash' => password_hash($payload['password'], PASSWORD_DEFAULT),
            'is_active' => $payload['is_active'],
            'created_at' => $now,
            'updated_at' => $now,
        ]);
        $this->aclModel->setUserRoles($userId, $payload['roles']);

        flash('success', 'تم إنشاء المستخدم بنجاح.');
        redirect('/users');
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));
        if ($id === '') {
            flash('error', 'معرف المستخدم غير صالح.');
            redirect('/users');
        }

        $user = $this->aclModel->findUserById($id);
        if ($user === null) {
            flash('error', 'المستخدم غير موجود.');
            redirect('/users');
        }

        $this->render('users/edit', [
            'title' => 'تعديل المستخدم',
            'user' => $user,
            'roles' => $this->aclModel->listRoles(true),
            'userRoleIds' => $this->aclModel->getUserRoleIds($id),
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validate($_POST, true, $id);

        if ($id === '') {
            $validated['errors'][] = 'معرف المستخدم غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/users/edit?id=' . urlencode($id));
        }

        $user = $this->aclModel->findUserById($id);
        if ($user === null) {
            flash('error', 'المستخدم غير موجود.');
            redirect('/users');
        }

        $payload = $validated['data'];
        $updatePassword = $payload['password'] !== '';

        $this->aclModel->updateUser([
            'id' => $id,
            'name' => $payload['name'],
            'email' => $payload['email'],
            'password_hash' => $updatePassword ? password_hash($payload['password'], PASSWORD_DEFAULT) : '',
            'is_active' => $payload['is_active'],
            'updated_at' => date('Y-m-d H:i:s'),
        ], $updatePassword);

        $this->aclModel->setUserRoles($id, $payload['roles']);

        if (AuthService::id() === $id && (int) $payload['is_active'] !== 1) {
            AuthService::logout();
            flash('success', 'تم تعطيل حسابك الحالي.');
            redirect('/login');
        }

        flash('success', 'تم تحديث المستخدم بنجاح.');
        redirect('/users');
    }

    public function toggleActive(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        if ($id === '') {
            flash('error', 'معرف المستخدم غير صالح.');
            redirect('/users');
        }

        $user = $this->aclModel->findUserById($id);
        if ($user === null) {
            flash('error', 'المستخدم غير موجود.');
            redirect('/users');
        }

        $newState = (int) $user['is_active'] === 1 ? 0 : 1;
        $this->aclModel->setUserActive($id, $newState);

        if (AuthService::id() === $id && $newState === 0) {
            AuthService::logout();
            flash('success', 'تم تعطيل حسابك الحالي.');
            redirect('/login');
        }

        flash('success', 'تم تحديث حالة المستخدم.');
        redirect('/users');
    }

    private function validate(array $input, bool $forUpdate, ?string $excludeId = null): array
    {
        $errors = [];

        $name = trim((string) ($input['name'] ?? ''));
        $email = strtolower(trim((string) ($input['email'] ?? '')));
        $password = (string) ($input['password'] ?? '');
        $isActive = (string) ($input['is_active'] ?? '1');
        $roles = $input['roles'] ?? [];
        $roles = is_array($roles) ? array_values(array_unique(array_map('strval', $roles))) : [];

        if ($name === '' || mb_strlen($name) > 120) {
            $errors[] = 'اسم المستخدم مطلوب وبحد أقصى 120 حرف.';
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'البريد الإلكتروني غير صالح.';
        } elseif ($this->aclModel->emailExists($email, $excludeId)) {
            $errors[] = 'البريد الإلكتروني مستخدم مسبقاً.';
        }

        if (!$forUpdate && strlen($password) < 8) {
            $errors[] = 'كلمة المرور مطلوبة وبحد أدنى 8 أحرف.';
        }

        if ($forUpdate && $password !== '' && strlen($password) < 8) {
            $errors[] = 'كلمة المرور الجديدة يجب أن تكون 8 أحرف على الأقل.';
        }

        if (!in_array($isActive, ['0', '1'], true)) {
            $errors[] = 'حالة المستخدم غير صحيحة.';
        }

        if ($roles === []) {
            $errors[] = 'يجب اختيار دور واحد على الأقل.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'name' => $name,
                'email' => $email,
                'password' => $password,
                'is_active' => (int) $isActive,
                'roles' => $roles,
            ],
        ];
    }
}
