<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\CategoryModel;
use App\Models\SettingsModel;

class SettingsController extends Controller
{
    private SettingsModel $settingsModel;
    private CategoryModel $categoryModel;

    public function __construct()
    {
        $this->settingsModel = new SettingsModel();
        $this->categoryModel = new CategoryModel();
    }

    public function index(): void
    {
        $settings = $this->settingsModel->getFirst();
        if ($settings === null) {
            flash('error', 'الإعدادات غير متاحة. شغّل الترحيلات أولاً.');
            redirect('/');
        }

        $this->render('settings/index', [
            'title' => 'الإعدادات',
            'settings' => $settings,
            'incomeCategories' => $this->categoryModel->listByType('income'),
            'expenseCategories' => $this->categoryModel->listByType('expense'),
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $settings = $this->settingsModel->getFirst();
        if ($settings === null) {
            flash('error', 'الإعدادات غير متاحة. شغّل الترحيلات أولاً.');
            redirect('/');
        }

        $validated = $this->validate($_POST);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/settings');
        }

        $payload = $validated['data'];
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->settingsModel->update((string) $settings['id'], $payload);

        flash('success', 'تم تحديث الإعدادات بنجاح.');
        redirect('/settings');
    }

    private function validate(array $input): array
    {
        $errors = [];

        $systemName = trim((string) ($input['system_name'] ?? ''));
        $currencyCode = strtoupper(trim((string) ($input['currency_code'] ?? '')));
        $numberFormat = trim((string) ($input['number_format'] ?? ''));
        $calendarType = trim((string) ($input['calendar_type'] ?? ''));
        $language = strtolower(trim((string) ($input['language'] ?? 'ar')));
        $rtl = (string) ($input['rtl'] ?? '1');

        if ($systemName === '' || mb_strlen($systemName) > 100) {
            $errors[] = 'اسم النظام مطلوب وبحد أقصى 100 حرف.';
        }

        if ($currencyCode === '' || mb_strlen($currencyCode) > 10) {
            $errors[] = 'رمز العملة مطلوب وبحد أقصى 10 أحرف.';
        }

        $allowedFormats = ['1,234.56', '1.234,56'];
        if (!in_array($numberFormat, $allowedFormats, true)) {
            $errors[] = 'صيغة الأرقام غير صحيحة.';
        }

        $allowedCalendarTypes = ['gregorian', 'hijri'];
        if (!in_array($calendarType, $allowedCalendarTypes, true)) {
            $errors[] = 'نوع التقويم غير صحيح.';
        }

        $allowedLanguages = ['ar', 'en'];
        if (!in_array($language, $allowedLanguages, true)) {
            $errors[] = 'قيمة اللغة غير صحيحة.';
        }

        if (!in_array($rtl, ['0', '1'], true)) {
            $errors[] = 'قيمة اتجاه الواجهة غير صحيحة.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'system_name' => $systemName,
                'currency_code' => $currencyCode,
                'number_format' => $numberFormat,
                'calendar_type' => $calendarType,
                'language' => $language,
                'rtl' => (int) $rtl,
            ],
        ];
    }
}
