<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\SentMoneyModel;
use App\Services\UuidService;

class SentMoneyController extends Controller
{
    private SentMoneyModel $model;
    private array $allowedStatuses = ['sent', 'received', 'overdue', 'cancelled'];

    public function __construct()
    {
        $this->model = new SentMoneyModel();
    }

    public function index(): void
    {
        $status = $this->resolveFilterStatus($_GET['status'] ?? 'all');

        $this->render('sent_money/index', [
            'title' => 'المبالغ المرسلة',
            'records' => $this->model->list($status),
            'statusFilter' => $status,
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validate($_POST);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/sent-money');
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $payload['id'] = UuidService::v4();
        $payload['created_at'] = $now;
        $payload['updated_at'] = $now;

        $this->model->create($payload);
        flash('success', 'تمت إضافة السجل بنجاح.');
        redirect('/sent-money');
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف السجل غير صالح.');
            redirect('/sent-money');
        }

        $record = $this->model->findById($id);
        if ($record === null) {
            flash('error', 'السجل غير موجود.');
            redirect('/sent-money');
        }

        $this->render('sent_money/edit', [
            'title' => 'تعديل مبلغ مرسل',
            'record' => $record,
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validate($_POST);

        if ($id === '') {
            $validated['errors'][] = 'معرف السجل غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/sent-money/edit?id=' . urlencode($id));
        }

        if ($this->model->findById($id) === null) {
            flash('error', 'السجل غير موجود.');
            redirect('/sent-money');
        }

        $payload = $validated['data'];
        $payload['id'] = $id;
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->model->update($payload);
        flash('success', 'تم تحديث السجل بنجاح.');
        redirect('/sent-money');
    }

    public function delete(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        if ($id === '') {
            flash('error', 'معرف السجل غير صالح.');
            redirect('/sent-money');
        }

        $this->model->delete($id);
        flash('success', 'تم حذف السجل بنجاح.');
        redirect('/sent-money');
    }

    private function validate(array $input): array
    {
        $errors = [];

        $personName = trim((string) ($input['person_name'] ?? ''));
        $amountRaw = str_replace(',', '', trim((string) ($input['amount'] ?? '')));
        $expectedReceiveDate = trim((string) ($input['expected_receive_date'] ?? ''));
        $status = trim((string) ($input['status'] ?? 'sent'));
        $notes = trim((string) ($input['notes'] ?? ''));
        $notes = $notes === '' ? null : $notes;

        if ($personName === '' || mb_strlen($personName) > 120) {
            $errors[] = 'اسم الشخص مطلوب وبحد أقصى 120 حرف.';
        }

        if ($amountRaw === '' || !is_numeric($amountRaw) || (float) $amountRaw <= 0) {
            $errors[] = 'قيمة المبلغ غير صحيحة.';
        }

        if (!$this->isValidDate($expectedReceiveDate)) {
            $errors[] = 'تاريخ الاستلام المتوقع غير صحيح (YYYY-MM-DD).';
        }

        if (!in_array($status, $this->allowedStatuses, true)) {
            $errors[] = 'حالة السجل غير صحيحة.';
        }

        if ($notes !== null && mb_strlen($notes) > 1000) {
            $errors[] = 'الملاحظات بحد أقصى 1000 حرف.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'person_name' => $personName,
                'amount' => (float) $amountRaw,
                'expected_receive_date' => $expectedReceiveDate,
                'status' => $status,
                'notes' => $notes,
            ],
        ];
    }

    private function resolveFilterStatus(mixed $status): string
    {
        $status = is_string($status) ? $status : 'all';
        if ($status === 'all') {
            return 'all';
        }

        return in_array($status, $this->allowedStatuses, true) ? $status : 'all';
    }

    private function isValidDate(string $date): bool
    {
        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
