<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\SearchService;

class SearchController extends Controller
{
    private SearchService $searchService;

    public function __construct()
    {
        $this->searchService = new SearchService();
    }

    public function index(): void
    {
        $keyword = trim((string) ($_GET['q'] ?? ''));
        $dateFrom = trim((string) ($_GET['date_from'] ?? ''));
        $dateTo = trim((string) ($_GET['date_to'] ?? ''));

        $errors = [];
        if ($dateFrom !== '' && !$this->isValidDate($dateFrom)) {
            $errors[] = 'صيغة تاريخ البداية غير صحيحة.';
            $dateFrom = '';
        }
        if ($dateTo !== '' && !$this->isValidDate($dateTo)) {
            $errors[] = 'صيغة تاريخ النهاية غير صحيحة.';
            $dateTo = '';
        }
        if ($dateFrom !== '' && $dateTo !== '' && $dateFrom > $dateTo) {
            $errors[] = 'تاريخ البداية يجب أن يكون قبل أو يساوي تاريخ النهاية.';
            $dateFrom = '';
            $dateTo = '';
        }

        $hasQuery = $keyword !== '' || $dateFrom !== '' || $dateTo !== '';
        $results = [
            'incomes' => [],
            'expenses' => [],
            'sent_money' => [],
            'borrowed_money' => [],
        ];

        if ($hasQuery && $errors === []) {
            $results = $this->searchService->search($keyword, $dateFrom, $dateTo);
        }

        $this->render('search/index', [
            'title' => 'البحث',
            'keyword' => $keyword,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'errors' => $errors,
            'hasQuery' => $hasQuery,
            'results' => $results,
        ]);
    }

    private function isValidDate(string $date): bool
    {
        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
