<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\AclModel;
use App\Services\UuidService;

class RoleController extends Controller
{
    private AclModel $aclModel;

    public function __construct()
    {
        $this->aclModel = new AclModel();
    }

    public function index(): void
    {
        $roles = $this->aclModel->listRoles();
        $permissions = $this->aclModel->listPermissions();

        $rolePermissions = [];
        foreach ($roles as $role) {
            $rolePermissions[(string) $role['id']] = $this->aclModel->getRolePermissionIds((string) $role['id']);
        }

        $this->render('roles/index', [
            'title' => 'إدارة الأدوار والصلاحيات',
            'roles' => $roles,
            'permissions' => $permissions,
            'rolePermissions' => $rolePermissions,
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $name = strtolower(trim((string) ($_POST['name'] ?? '')));
        $displayName = trim((string) ($_POST['display_name'] ?? ''));

        $errors = [];
        if ($name === '' || !preg_match('/^[a-z0-9._-]{2,50}$/', $name)) {
            $errors[] = 'اسم الدور يجب أن يكون 2-50 حرفًا (a-z, 0-9, . _ -).';
        } elseif ($this->aclModel->roleNameExists($name)) {
            $errors[] = 'اسم الدور مستخدم مسبقاً.';
        }

        if ($displayName === '' || mb_strlen($displayName) > 120) {
            $errors[] = 'الاسم المعروض مطلوب وبحد أقصى 120 حرف.';
        }

        if ($errors !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $errors));
            redirect('/roles');
        }

        $now = date('Y-m-d H:i:s');
        $this->aclModel->createRole([
            'id' => UuidService::v4(),
            'name' => $name,
            'display_name' => $displayName,
            'is_active' => 1,
            'created_at' => $now,
            'updated_at' => $now,
        ]);

        flash('success', 'تم إنشاء الدور بنجاح.');
        redirect('/roles');
    }

    public function updateRole(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $displayName = trim((string) ($_POST['display_name'] ?? ''));
        $isActive = (string) ($_POST['is_active'] ?? '1');

        $errors = [];
        if ($id === '') {
            $errors[] = 'معرف الدور غير صالح.';
        }

        if ($displayName === '' || mb_strlen($displayName) > 120) {
            $errors[] = 'الاسم المعروض مطلوب وبحد أقصى 120 حرف.';
        }

        if (!in_array($isActive, ['0', '1'], true)) {
            $errors[] = 'حالة الدور غير صحيحة.';
        }

        if ($errors !== []) {
            flash('error', implode(' ', $errors));
            redirect('/roles');
        }

        if ($this->aclModel->findRoleById($id) === null) {
            flash('error', 'الدور غير موجود.');
            redirect('/roles');
        }

        $this->aclModel->updateRole([
            'id' => $id,
            'display_name' => $displayName,
            'is_active' => (int) $isActive,
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        flash('success', 'تم تحديث بيانات الدور.');
        redirect('/roles');
    }

    public function updatePermissions(): void
    {
        $roleId = trim((string) ($_POST['role_id'] ?? ''));
        $selectedPermissions = $_POST['permissions'] ?? [];
        $selectedPermissions = is_array($selectedPermissions)
            ? array_values(array_unique(array_map('strval', $selectedPermissions)))
            : [];

        if ($roleId === '') {
            flash('error', 'معرف الدور غير صالح.');
            redirect('/roles');
        }

        $role = $this->aclModel->findRoleById($roleId);
        if ($role === null) {
            flash('error', 'الدور غير موجود.');
            redirect('/roles');
        }

        $validPermissionIds = $this->aclModel->listPermissionIds();
        $validSet = array_flip($validPermissionIds);
        foreach ($selectedPermissions as $permissionId) {
            if (!isset($validSet[$permissionId])) {
                flash('error', 'تم إرسال صلاحيات غير معروفة.');
                redirect('/roles');
            }
        }

        $this->aclModel->setRolePermissions($roleId, $selectedPermissions);
        flash('success', 'تم تحديث صلاحيات الدور.');
        redirect('/roles');
    }
}
