<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\ReportService;

class ReportController extends Controller
{
    private ReportService $service;

    public function __construct()
    {
        $this->service = new ReportService();
    }

    public function index(): void
    {
        $month = trim((string) ($_GET['month'] ?? date('Y-m')));
        if (!$this->isValidMonth($month)) {
            flash('error', 'صيغة الشهر غير صحيحة. استخدم YYYY-MM.');
            redirect('/reports');
        }

        $summary = $this->service->monthlySummary($month);

        $this->render('reports/index', [
            'title' => 'التقارير الشهرية',
            'month' => $month,
            'summary' => $summary,
        ]);
    }

    public function exportJson(): void
    {
        $month = trim((string) ($_GET['month'] ?? date('Y-m')));
        if (!$this->isValidMonth($month)) {
            http_response_code(422);
            echo 'Invalid month format.';
            return;
        }

        $summary = $this->service->monthlySummary($month);
        $fileName = 'report-' . $month . '.json';

        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        echo json_encode($summary, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }

    private function isValidMonth(string $month): bool
    {
        return preg_match('/^\d{4}-\d{2}$/', $month) === 1;
    }
}
