<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\ReminderModel;
use App\Services\UuidService;

class ReminderController extends Controller
{
    private ReminderModel $model;
    private array $allowedStatuses = ['pending', 'confirmed', 'postponed', 'cancelled'];

    public function __construct()
    {
        $this->model = new ReminderModel();
    }

    public function index(): void
    {
        $status = $this->resolveFilterStatus($_GET['status'] ?? 'all');

        $this->render('reminders/index', [
            'title' => 'التذكيرات الشهرية',
            'records' => $this->model->list($status),
            'statusFilter' => $status,
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validate($_POST);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/reminders');
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $payload['id'] = UuidService::v4();
        $payload['created_at'] = $now;
        $payload['updated_at'] = $now;

        $this->model->create($payload);
        flash('success', 'تمت إضافة التذكير بنجاح.');
        redirect('/reminders');
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف التذكير غير صالح.');
            redirect('/reminders');
        }

        $record = $this->model->findById($id);
        if ($record === null) {
            flash('error', 'التذكير غير موجود.');
            redirect('/reminders');
        }

        $this->render('reminders/edit', [
            'title' => 'تعديل التذكير',
            'record' => $record,
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validate($_POST);

        if ($id === '') {
            $validated['errors'][] = 'معرف التذكير غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/reminders/edit?id=' . urlencode($id));
        }

        if ($this->model->findById($id) === null) {
            flash('error', 'التذكير غير موجود.');
            redirect('/reminders');
        }

        $payload = $validated['data'];
        $payload['id'] = $id;
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->model->update($payload);
        flash('success', 'تم تحديث التذكير بنجاح.');
        redirect('/reminders');
    }

    public function delete(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        if ($id === '') {
            flash('error', 'معرف التذكير غير صالح.');
            redirect('/reminders');
        }

        $this->model->delete($id);
        flash('success', 'تم حذف التذكير بنجاح.');
        redirect('/reminders');
    }

    private function validate(array $input): array
    {
        $errors = [];

        $title = trim((string) ($input['title'] ?? ''));
        $amountRaw = trim((string) ($input['amount'] ?? ''));
        $dueDate = trim((string) ($input['due_date'] ?? ''));
        $status = trim((string) ($input['status'] ?? 'pending'));
        $notes = trim((string) ($input['notes'] ?? ''));
        $notes = $notes === '' ? null : $notes;

        $amount = null;
        if ($amountRaw !== '') {
            $amountRaw = str_replace(',', '', $amountRaw);
            if (!is_numeric($amountRaw) || (float) $amountRaw < 0) {
                $errors[] = 'قيمة المبلغ غير صحيحة.';
            } else {
                $amount = (float) $amountRaw;
            }
        }

        if ($title === '' || mb_strlen($title) > 160) {
            $errors[] = 'عنوان التذكير مطلوب وبحد أقصى 160 حرف.';
        }

        if (!$this->isValidDate($dueDate)) {
            $errors[] = 'تاريخ الاستحقاق غير صحيح (YYYY-MM-DD).';
        }

        if (!in_array($status, $this->allowedStatuses, true)) {
            $errors[] = 'حالة التذكير غير صحيحة.';
        }

        if ($notes !== null && mb_strlen($notes) > 1000) {
            $errors[] = 'الملاحظات بحد أقصى 1000 حرف.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'title' => $title,
                'amount' => $amount,
                'due_date' => $dueDate,
                'status' => $status,
                'notes' => $notes,
            ],
        ];
    }

    private function resolveFilterStatus(mixed $status): string
    {
        $status = is_string($status) ? $status : 'all';
        if ($status === 'all') {
            return 'all';
        }

        return in_array($status, $this->allowedStatuses, true) ? $status : 'all';
    }

    private function isValidDate(string $date): bool
    {
        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
