<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\CategoryModel;
use App\Models\RecurringModel;
use App\Services\UuidService;

class RecurringController extends Controller
{
    private RecurringModel $model;
    private CategoryModel $categoryModel;

    public function __construct()
    {
        $this->model = new RecurringModel();
        $this->categoryModel = new CategoryModel();
    }

    public function index(): void
    {
        $this->renderIndex($this->resolveType($_GET['type'] ?? 'income'));
    }

    public function incomes(): void
    {
        $this->renderIndex('income');
    }

    public function expenses(): void
    {
        $this->renderIndex('expense');
    }

    public function store(): void
    {
        $validated = $this->validate($_POST, false);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect($this->listPath($validated['type']));
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $payload['id'] = UuidService::v4();
        $payload['is_active'] = 1;
        $payload['created_at'] = $now;
        $payload['updated_at'] = $now;

        $this->model->create($validated['type'], $payload);
        flash('success', 'تمت إضافة السجل المتكرر بنجاح.');
        redirect($this->listPath($validated['type']));
    }

    public function edit(): void
    {
        $this->renderEdit($this->resolveType($_GET['type'] ?? 'income'));
    }

    public function editIncome(): void
    {
        $this->renderEdit('income');
    }

    public function editExpense(): void
    {
        $this->renderEdit('expense');
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validate($_POST, true);

        if ($id === '') {
            $validated['errors'][] = 'معرف السجل غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect($this->editPath($validated['type'], $id));
        }

        if ($this->model->findById($validated['type'], $id) === null) {
            flash('error', 'السجل غير موجود.');
            redirect($this->listPath($validated['type']));
        }

        $payload = $validated['data'];
        $payload['id'] = $id;
        $payload['updated_at'] = date('Y-m-d H:i:s');
        $payload['is_active'] = $validated['is_active'];

        $this->model->update($validated['type'], $payload);
        flash('success', 'تم تحديث السجل المتكرر بنجاح.');
        redirect($this->listPath($validated['type']));
    }

    public function delete(): void
    {
        $type = $this->resolveType($_POST['type'] ?? 'income');
        $id = trim((string) ($_POST['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف السجل غير صالح.');
            redirect($this->listPath($type));
        }

        $this->model->delete($type, $id);
        flash('success', 'تم حذف السجل المتكرر بنجاح.');
        redirect($this->listPath($type));
    }

    public function generate(): void
    {
        $type = $this->resolveType($_POST['type'] ?? 'income');

        $confirm = trim((string) ($_POST['confirm'] ?? ''));
        if ($confirm !== 'yes') {
            flash('error', 'يجب تأكيد عملية التوليد أولاً.');
            redirect($this->listPath($type));
        }

        $result = $this->model->generateCurrentMonthForType($type);
        $monthLabel = sprintf('%04d-%02d', $result['year'], $result['month']);
        $typeLabel = $type === 'expense' ? 'المصروفات' : 'الإيرادات';

        flash(
            'success',
            'تم توليد ' . $typeLabel . ' للشهر ' . $monthLabel . ' بنجاح. '
            . 'تم الإنشاء: ' . $result['created'] . ' | تم التخطي: ' . $result['skipped']
        );
        redirect($this->listPath($type));
    }

    public function generateSingle(): void
    {
        $type = $this->resolveType($_POST['type'] ?? 'income');
        $id = trim((string) ($_POST['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف السجل غير صالح.');
            redirect($this->listPath($type));
        }

        $record = $this->model->findById($type, $id);
        if ($record === null) {
            flash('error', 'السجل غير موجود.');
            redirect($this->listPath($type));
        }

        if ((int) ($record['is_active'] ?? 0) !== 1) {
            flash('error', 'لا يمكن التوليد لسجل غير نشط.');
            redirect($this->listPath($type));
        }

        $result = $this->model->generateSingleCurrentMonth($type, $id);
        $monthLabel = sprintf('%04d-%02d', $result['year'], $result['month']);

        if ($result['created'] > 0) {
            flash('success', 'تم توليد السجل بنجاح للشهر ' . $monthLabel . '.');
            redirect($this->listPath($type));
        }

        flash('error', 'هذا السجل تم توليده مسبقًا لنفس الشهر ' . $monthLabel . '.');
        redirect($this->listPath($type));
    }

    private function validate(array $input, bool $forUpdate): array
    {
        $errors = [];

        $type = $this->resolveType($input['type'] ?? 'income');
        $amountRaw = str_replace(',', '', trim((string) ($input['amount'] ?? '')));
        $categoryIdRaw = trim((string) ($input['category_id'] ?? ''));
        $description = trim((string) ($input['description'] ?? ''));
        $dayRaw = trim((string) ($input['day_of_month'] ?? ''));
        $isActive = trim((string) ($input['is_active'] ?? '1'));
        $categoryId = $categoryIdRaw === '' ? null : $categoryIdRaw;

        if ($amountRaw === '' || !is_numeric($amountRaw) || (float) $amountRaw <= 0) {
            $errors[] = 'قيمة المبلغ غير صحيحة.';
        }

        if ($description === '' || mb_strlen($description) > 500) {
            $errors[] = 'الوصف مطلوب وبحد أقصى 500 حرف.';
        }

        if ($dayRaw === '' || !ctype_digit($dayRaw) || (int) $dayRaw < 1 || (int) $dayRaw > 31) {
            $errors[] = 'يوم الشهر يجب أن يكون بين 1 و31.';
        }

        if ($categoryId !== null && !$this->categoryModel->existsByType($categoryId, $type)) {
            $errors[] = 'التصنيف المحدد غير صالح لهذا النوع.';
        }

        if ($forUpdate && !in_array($isActive, ['0', '1'], true)) {
            $errors[] = 'حالة التفعيل غير صحيحة.';
        }

        return [
            'errors' => $errors,
            'type' => $type,
            'is_active' => (int) $isActive,
            'data' => [
                'amount' => (float) $amountRaw,
                'category_id' => $categoryId,
                'description' => $description,
                'day_of_month' => (int) $dayRaw,
            ],
        ];
    }

    private function resolveType(mixed $type): string
    {
        return $type === 'expense' ? 'expense' : 'income';
    }

    private function renderIndex(string $type): void
    {
        $title = $type === 'expense' ? 'المصروفات المتكررة' : 'الإيرادات المتكررة';

        $this->render('recurring/index', [
            'title' => $title,
            'type' => $type,
            'records' => $this->model->list($type),
            'categories' => $this->categoryModel->listByType($type),
            'oldInput' => get_old_input(),
        ]);
    }

    private function renderEdit(string $type): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));
        if ($id === '') {
            flash('error', 'معرف السجل غير صالح.');
            redirect($this->listPath($type));
        }

        $record = $this->model->findById($type, $id);
        if ($record === null) {
            flash('error', 'السجل غير موجود.');
            redirect($this->listPath($type));
        }

        $title = $type === 'expense' ? 'تعديل مصروف متكرر' : 'تعديل إيراد متكرر';

        $this->render('recurring/edit', [
            'title' => $title,
            'type' => $type,
            'record' => $record,
            'categories' => $this->categoryModel->listByType($type),
            'oldInput' => get_old_input(),
        ]);
    }

    private function listPath(string $type): string
    {
        return $type === 'expense' ? '/recurring/expenses' : '/recurring/incomes';
    }

    private function editPath(string $type, string $id): string
    {
        $base = $type === 'expense' ? '/recurring/expenses/edit' : '/recurring/incomes/edit';
        return $base . '?id=' . urlencode($id);
    }
}
