<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\CategoryModel;
use App\Models\IncomeModel;
use App\Services\UuidService;

class IncomeController extends Controller
{
    private IncomeModel $incomeModel;
    private CategoryModel $categoryModel;

    public function __construct()
    {
        $this->incomeModel = new IncomeModel();
        $this->categoryModel = new CategoryModel();
    }

    public function index(): void
    {
        $filters = $this->validatedFilters($_GET);

        $this->render('incomes/index', [
            'title' => 'الإيرادات',
            'filters' => $filters['data'],
            'filterErrors' => $filters['errors'],
            'incomes' => $this->incomeModel->list($filters['data']),
            'summary' => $this->incomeModel->summary($filters['data']),
            'categories' => $this->categoryModel->listByType('income'),
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validateIncome($_POST);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/incomes');
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $payload['id'] = UuidService::v4();
        $payload['created_at'] = $now;
        $payload['updated_at'] = $now;

        $this->incomeModel->create($payload);

        flash('success', 'تمت إضافة الإيراد بنجاح.');
        redirect('/incomes');
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف الإيراد غير صالح.');
            redirect('/incomes');
        }

        $income = $this->incomeModel->findById($id);

        if ($income === null) {
            flash('error', 'الإيراد غير موجود.');
            redirect('/incomes');
        }

        $this->render('incomes/edit', [
            'title' => 'تعديل الإيراد',
            'income' => $income,
            'categories' => $this->categoryModel->listByType('income'),
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validateIncome($_POST);

        if ($id === '') {
            $validated['errors'][] = 'معرف الإيراد غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/incomes/edit?id=' . urlencode($id));
        }

        if ($this->incomeModel->findById($id) === null) {
            flash('error', 'الإيراد غير موجود.');
            redirect('/incomes');
        }

        $payload = $validated['data'];
        $payload['id'] = $id;
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->incomeModel->update($payload);

        flash('success', 'تم تحديث الإيراد بنجاح.');
        redirect('/incomes');
    }

    public function delete(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف الإيراد غير صالح.');
            redirect('/incomes');
        }

        $this->incomeModel->delete($id);

        flash('success', 'تم حذف الإيراد بنجاح.');
        redirect('/incomes');
    }

    private function validateIncome(array $input): array
    {
        $errors = [];

        $amountRaw = trim((string) ($input['amount'] ?? ''));
        $amountRaw = str_replace(',', '', $amountRaw);
        $description = trim((string) ($input['description'] ?? ''));
        $date = trim((string) ($input['date'] ?? ''));
        $categoryIdRaw = trim((string) ($input['category_id'] ?? ''));
        $categoryId = $categoryIdRaw === '' ? null : $categoryIdRaw;

        if ($amountRaw === '' || !is_numeric($amountRaw)) {
            $errors[] = 'قيمة المبلغ غير صحيحة.';
        }

        $amount = (float) $amountRaw;
        if ($amount <= 0) {
            $errors[] = 'قيمة الإيراد يجب أن تكون أكبر من صفر.';
        }

        if ($description === '' || mb_strlen($description) > 500) {
            $errors[] = 'الوصف مطلوب وبحد أقصى 500 حرف.';
        }

        if (!$this->isValidDate($date)) {
            $errors[] = 'تاريخ الإيراد غير صحيح (YYYY-MM-DD).';
        }

        if ($categoryId !== null && !$this->categoryModel->existsByType($categoryId, 'income')) {
            $errors[] = 'التصنيف المحدد غير صالح للإيرادات.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'amount' => $amount,
                'category_id' => $categoryId,
                'description' => $description,
                'date' => $date,
            ],
        ];
    }

    private function validatedFilters(array $input): array
    {
        $errors = [];

        $dateFrom = trim((string) ($input['date_from'] ?? ''));
        $dateTo = trim((string) ($input['date_to'] ?? ''));

        if ($dateFrom !== '' && !$this->isValidDate($dateFrom)) {
            $errors[] = 'صيغة تاريخ البداية غير صحيحة.';
            $dateFrom = '';
        }

        if ($dateTo !== '' && !$this->isValidDate($dateTo)) {
            $errors[] = 'صيغة تاريخ النهاية غير صحيحة.';
            $dateTo = '';
        }

        if ($dateFrom !== '' && $dateTo !== '' && $dateFrom > $dateTo) {
            $errors[] = 'تاريخ البداية يجب أن يكون قبل أو يساوي تاريخ النهاية.';
            $dateFrom = '';
            $dateTo = '';
        }

        return [
            'errors' => $errors,
            'data' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
        ];
    }

    private function isValidDate(string $date): bool
    {
        if ($date === '') {
            return false;
        }

        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
