<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Database\Connection;
use App\Services\DashboardService;
use Throwable;

class HomeController extends Controller
{
    public function index(): void
    {
        $filters = $this->validatedFilters($_GET);

        try {
            $dashboard = new DashboardService();
            $stats = $dashboard->stats($filters['data']);
        } catch (Throwable $exception) {
            flash('error', 'لا يمكن تحميل لوحة التحكم حالياً. شغّل الترحيلات أولاً.');
            redirect('/health');
        }

        $this->render('home/index', [
            'title' => 'لوحة التحكم',
            'stats' => $stats,
            'filters' => $filters['data'],
            'filterErrors' => $filters['errors'],
        ]);
    }

    public function health(): void
    {
        $dbStatus = 'متصل';

        try {
            $pdo = Connection::getInstance();
            $pdo->query('SELECT 1');
        } catch (Throwable $exception) {
            $dbStatus = 'غير متصل: ' . $exception->getMessage();
        }

        $this->render('home/health', [
            'title' => 'فحص النظام',
            'dbStatus' => $dbStatus,
        ]);
    }

    private function validatedFilters(array $input): array
    {
        $errors = [];

        $dateFrom = trim((string) ($input['date_from'] ?? ''));
        $dateTo = trim((string) ($input['date_to'] ?? ''));

        if ($dateFrom !== '' && !$this->isValidDate($dateFrom)) {
            $errors[] = 'صيغة تاريخ البداية غير صحيحة.';
            $dateFrom = '';
        }

        if ($dateTo !== '' && !$this->isValidDate($dateTo)) {
            $errors[] = 'صيغة تاريخ النهاية غير صحيحة.';
            $dateTo = '';
        }

        if ($dateFrom !== '' && $dateTo !== '' && $dateFrom > $dateTo) {
            $errors[] = 'تاريخ البداية يجب أن يكون قبل أو يساوي تاريخ النهاية.';
            $dateFrom = '';
            $dateTo = '';
        }

        return [
            'errors' => $errors,
            'data' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
        ];
    }

    private function isValidDate(string $date): bool
    {
        if ($date === '') {
            return false;
        }

        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
