<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\CategoryModel;
use App\Models\ExpenseModel;
use App\Services\UuidService;

class ExpenseController extends Controller
{
    private ExpenseModel $expenseModel;
    private CategoryModel $categoryModel;

    public function __construct()
    {
        $this->expenseModel = new ExpenseModel();
        $this->categoryModel = new CategoryModel();
    }

    public function index(): void
    {
        $filters = $this->validatedFilters($_GET);

        $this->render('expenses/index', [
            'title' => 'المصروفات',
            'filters' => $filters['data'],
            'filterErrors' => $filters['errors'],
            'expenses' => $this->expenseModel->list($filters['data']),
            'summary' => $this->expenseModel->summary($filters['data']),
            'categories' => $this->categoryModel->listByType('expense'),
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validateExpense($_POST);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/expenses');
        }

        $now = date('Y-m-d H:i:s');
        $payload = $validated['data'];
        $payload['id'] = UuidService::v4();
        $payload['created_at'] = $now;
        $payload['updated_at'] = $now;

        $this->expenseModel->create($payload);

        flash('success', 'تمت إضافة المصروف بنجاح.');
        redirect('/expenses');
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف المصروف غير صالح.');
            redirect('/expenses');
        }

        $expense = $this->expenseModel->findById($id);

        if ($expense === null) {
            flash('error', 'المصروف غير موجود.');
            redirect('/expenses');
        }

        $this->render('expenses/edit', [
            'title' => 'تعديل المصروف',
            'expense' => $expense,
            'categories' => $this->categoryModel->listByType('expense'),
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validateExpense($_POST);

        if ($id === '') {
            $validated['errors'][] = 'معرف المصروف غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect('/expenses/edit?id=' . urlencode($id));
        }

        if ($this->expenseModel->findById($id) === null) {
            flash('error', 'المصروف غير موجود.');
            redirect('/expenses');
        }

        $payload = $validated['data'];
        $payload['id'] = $id;
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->expenseModel->update($payload);

        flash('success', 'تم تحديث المصروف بنجاح.');
        redirect('/expenses');
    }

    public function delete(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف المصروف غير صالح.');
            redirect('/expenses');
        }

        $this->expenseModel->delete($id);

        flash('success', 'تم حذف المصروف بنجاح.');
        redirect('/expenses');
    }

    private function validateExpense(array $input): array
    {
        $errors = [];

        $amountRaw = str_replace(',', '', trim((string) ($input['amount'] ?? '')));
        $description = trim((string) ($input['description'] ?? ''));
        $date = trim((string) ($input['date'] ?? ''));
        $categoryIdRaw = trim((string) ($input['category_id'] ?? ''));
        $categoryId = $categoryIdRaw === '' ? null : $categoryIdRaw;

        if ($amountRaw === '' || !is_numeric($amountRaw)) {
            $errors[] = 'قيمة المبلغ غير صحيحة.';
        }

        $amount = (float) $amountRaw;
        if ($amount <= 0) {
            $errors[] = 'قيمة المصروف يجب أن تكون أكبر من صفر.';
        }

        if ($description === '' || mb_strlen($description) > 500) {
            $errors[] = 'الوصف مطلوب وبحد أقصى 500 حرف.';
        }

        if (!$this->isValidDate($date)) {
            $errors[] = 'تاريخ المصروف غير صحيح (YYYY-MM-DD).';
        }

        if ($categoryId !== null && !$this->categoryModel->existsByType($categoryId, 'expense')) {
            $errors[] = 'التصنيف المحدد غير صالح للمصروفات.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'amount' => $amount,
                'category_id' => $categoryId,
                'description' => $description,
                'date' => $date,
            ],
        ];
    }

    private function validatedFilters(array $input): array
    {
        $errors = [];

        $dateFrom = trim((string) ($input['date_from'] ?? ''));
        $dateTo = trim((string) ($input['date_to'] ?? ''));

        if ($dateFrom !== '' && !$this->isValidDate($dateFrom)) {
            $errors[] = 'صيغة تاريخ البداية غير صحيحة.';
            $dateFrom = '';
        }

        if ($dateTo !== '' && !$this->isValidDate($dateTo)) {
            $errors[] = 'صيغة تاريخ النهاية غير صحيحة.';
            $dateTo = '';
        }

        if ($dateFrom !== '' && $dateTo !== '' && $dateFrom > $dateTo) {
            $errors[] = 'تاريخ البداية يجب أن يكون قبل أو يساوي تاريخ النهاية.';
            $dateFrom = '';
            $dateTo = '';
        }

        return [
            'errors' => $errors,
            'data' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
        ];
    }

    private function isValidDate(string $date): bool
    {
        if ($date === '') {
            return false;
        }

        $dateTime = \DateTime::createFromFormat('Y-m-d', $date);
        return $dateTime instanceof \DateTime && $dateTime->format('Y-m-d') === $date;
    }
}
