<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\CategoryModel;
use App\Services\UuidService;

class CategoryController extends Controller
{
    private CategoryModel $categoryModel;

    public function __construct()
    {
        $this->categoryModel = new CategoryModel();
    }

    public function index(): void
    {
        $type = $this->resolveType($_GET['type'] ?? 'income');

        $this->render('categories/index', [
            'title' => 'إدارة التصنيفات',
            'type' => $type,
            'categories' => $this->categoryModel->listByType($type),
            'oldInput' => get_old_input(),
        ]);
    }

    public function store(): void
    {
        $validated = $this->validate($_POST, false);
        $type = $this->resolveType($_POST['type'] ?? 'income');
        $returnTo = $this->resolveReturnTo($_POST, '/categories?type=' . $type);

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect($returnTo);
        }

        $now = date('Y-m-d H:i:s');
        $this->categoryModel->create([
            'id' => UuidService::v4(),
            'type' => $validated['data']['type'],
            'name' => $validated['data']['name'],
            'is_active' => 1,
            'created_at' => $now,
            'updated_at' => $now,
        ]);

        flash('success', 'تمت إضافة التصنيف بنجاح.');
        redirect($returnTo);
    }

    public function edit(): void
    {
        $id = trim((string) ($_GET['id'] ?? ''));

        if ($id === '') {
            flash('error', 'معرف التصنيف غير صالح.');
            redirect('/categories');
        }

        $category = $this->categoryModel->findById($id);

        if ($category === null) {
            flash('error', 'التصنيف غير موجود.');
            redirect('/categories');
        }

        $this->render('categories/edit', [
            'title' => 'تعديل التصنيف',
            'category' => $category,
            'oldInput' => get_old_input(),
        ]);
    }

    public function update(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $validated = $this->validate($_POST, true);
        $returnTo = $this->resolveReturnTo($_POST, '/categories/edit?id=' . urlencode($id));

        if ($id === '') {
            $validated['errors'][] = 'معرف التصنيف غير صالح.';
        }

        if ($validated['errors'] !== []) {
            set_old_input($_POST);
            flash('error', implode(' ', $validated['errors']));
            redirect($returnTo);
        }

        $updated = $this->categoryModel->update([
            'id' => $id,
            'type' => $validated['data']['type'],
            'name' => $validated['data']['name'],
            'is_active' => $validated['data']['is_active'],
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        if (!$updated) {
            flash('error', 'تعذر تحديث التصنيف.');
            redirect($returnTo);
        }

        flash('success', 'تم تحديث التصنيف بنجاح.');
        redirect($this->resolveReturnTo($_POST, '/categories?type=' . $validated['data']['type']));
    }

    public function deactivate(): void
    {
        $id = trim((string) ($_POST['id'] ?? ''));
        $type = $this->resolveType($_POST['type'] ?? 'income');
        $returnTo = $this->resolveReturnTo($_POST, '/categories?type=' . $type);

        if ($id === '') {
            flash('error', 'معرف التصنيف غير صالح.');
            redirect($returnTo);
        }

        $updated = $this->categoryModel->deactivate($id, date('Y-m-d H:i:s'));

        if (!$updated) {
            flash('error', 'تعذر تعطيل التصنيف.');
            redirect($returnTo);
        }

        flash('success', 'تم تعطيل التصنيف بنجاح.');
        redirect($returnTo);
    }

    private function validate(array $input, bool $forUpdate): array
    {
        $errors = [];

        $rawType = (string) ($input['type'] ?? '');
        $type = $this->resolveType($rawType);
        $name = trim((string) ($input['name'] ?? ''));
        $isActive = (string) ($input['is_active'] ?? '1');

        if (!in_array($rawType, ['income', 'expense'], true)) {
            $errors[] = 'نوع التصنيف غير صحيح.';
        }

        if ($name === '' || mb_strlen($name) > 100) {
            $errors[] = 'اسم التصنيف مطلوب وبحد أقصى 100 حرف.';
        }

        if ($forUpdate && !in_array($isActive, ['0', '1'], true)) {
            $errors[] = 'حالة التصنيف غير صحيحة.';
        }

        return [
            'errors' => $errors,
            'data' => [
                'type' => $type,
                'name' => $name,
                'is_active' => (int) $isActive,
            ],
        ];
    }

    private function resolveType(mixed $type): string
    {
        return $type === 'expense' ? 'expense' : 'income';
    }

    private function resolveReturnTo(array $input, string $default): string
    {
        $returnTo = trim((string) ($input['return_to'] ?? ''));
        if ($returnTo === '') {
            return $default;
        }

        if (!str_starts_with($returnTo, '/')) {
            return $default;
        }

        if (str_contains($returnTo, "\r") || str_contains($returnTo, "\n")) {
            return $default;
        }

        return $returnTo;
    }
}
