<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Services\BackupService;
use RuntimeException;

class BackupController extends Controller
{
    private BackupService $service;

    public function __construct()
    {
        $this->service = new BackupService();
    }

    public function index(): void
    {
        $this->render('backup/index', [
            'title' => 'النسخ الاحتياطي والاستعادة',
        ]);
    }

    public function downloadDatabase(): void
    {
        $path = $this->service->databasePath();
        if (!is_file($path)) {
            http_response_code(404);
            echo 'Database not found.';
            return;
        }

        $fileName = 'expense-income-' . date('Ymd-His') . '.sqlite';
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        header('Content-Length: ' . (string) filesize($path));
        readfile($path);
    }

    public function exportJson(): void
    {
        $payload = $this->service->exportJsonData();
        $fileName = 'expense-income-export-' . date('Ymd-His') . '.json';

        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }

    public function restoreSqlite(): void
    {
        $confirm = trim((string) ($_POST['confirm_restore'] ?? ''));
        if ($confirm !== 'RESTORE') {
            flash('error', 'يجب كتابة RESTORE للتأكيد.');
            redirect('/backup');
        }

        if (!isset($_FILES['sqlite_file']) || !is_array($_FILES['sqlite_file'])) {
            flash('error', 'يرجى رفع ملف SQLite صالح.');
            redirect('/backup');
        }

        $tmpPath = (string) ($_FILES['sqlite_file']['tmp_name'] ?? '');
        $originalName = (string) ($_FILES['sqlite_file']['name'] ?? '');
        if ($tmpPath === '' || !is_file($tmpPath) || !str_ends_with(strtolower($originalName), '.sqlite')) {
            flash('error', 'يجب أن يكون الملف بامتداد .sqlite');
            redirect('/backup');
        }

        try {
            $this->service->restoreDatabaseFromFile($tmpPath);
            flash('success', 'تمت استعادة قاعدة البيانات بنجاح.');
        } catch (RuntimeException $e) {
            flash('error', $e->getMessage());
        }

        redirect('/backup');
    }

    public function importJson(): void
    {
        $confirm = trim((string) ($_POST['confirm_import'] ?? ''));
        if ($confirm !== 'REPLACE') {
            flash('error', 'يجب كتابة REPLACE لتأكيد الاستبدال الكامل.');
            redirect('/backup');
        }

        if (!isset($_FILES['json_file']) || !is_array($_FILES['json_file'])) {
            flash('error', 'يرجى رفع ملف JSON صالح.');
            redirect('/backup');
        }

        $tmpPath = (string) ($_FILES['json_file']['tmp_name'] ?? '');
        if ($tmpPath === '' || !is_file($tmpPath)) {
            flash('error', 'تعذر قراءة ملف JSON.');
            redirect('/backup');
        }

        $content = file_get_contents($tmpPath);
        if ($content === false) {
            flash('error', 'تعذر قراءة محتوى ملف JSON.');
            redirect('/backup');
        }

        try {
            $this->service->importJsonReplace($content);
            flash('success', 'تم استيراد JSON بنجاح (وضع الاستبدال الكامل).');
        } catch (\Throwable $e) {
            flash('error', 'فشل الاستيراد: ' . $e->getMessage());
        }

        redirect('/backup');
    }

    public function factoryReset(): void
    {
        $confirm = trim((string) ($_POST['confirm_factory_reset'] ?? ''));
        if ($confirm !== 'RESET') {
            flash('error', 'يجب كتابة RESET لتأكيد إعادة الضبط.');
            redirect('/backup');
        }

        try {
            $this->service->factoryReset();
            flash('success', 'تم تنفيذ Factory Reset وإعادة التهيئة بنجاح.');
        } catch (\Throwable $e) {
            flash('error', 'فشل Factory Reset: ' . $e->getMessage());
        }

        redirect('/backup');
    }
}
