<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\AclModel;
use App\Services\AuthService;
use App\Services\UuidService;

class AuthController extends Controller
{
    private AclModel $aclModel;

    public function __construct()
    {
        $this->aclModel = new AclModel();
    }

    public function showLogin(): void
    {
        if (AuthService::check()) {
            redirect('/');
        }

        $this->render('auth/login', [
            'title' => 'تسجيل الدخول',
            'oldInput' => get_old_input(),
        ]);
    }

    public function showRegister(): void
    {
        if (AuthService::check()) {
            redirect('/');
        }

        $this->render('auth/register', [
            'title' => 'إنشاء حساب جديد',
            'oldInput' => get_old_input(),
        ]);
    }

    public function login(): void
    {
        $email = strtolower(trim((string) ($_POST['email'] ?? '')));
        $password = (string) ($_POST['password'] ?? '');

        if ($email === '' || $password === '') {
            set_old_input(['email' => $email]);
            flash('error', 'البريد الإلكتروني وكلمة المرور مطلوبان.');
            redirect('/login');
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            set_old_input(['email' => $email]);
            flash('error', 'صيغة البريد الإلكتروني غير صحيحة.');
            redirect('/login');
        }

        if (!AuthService::login($email, $password)) {
            set_old_input(['email' => $email]);
            flash('error', 'بيانات الدخول غير صحيحة أو الحساب غير نشط.');
            redirect('/login');
        }

        flash('success', 'تم تسجيل الدخول بنجاح.');
        redirect('/');
    }

    public function register(): void
    {
        if (AuthService::check()) {
            redirect('/');
        }

        $name = trim((string) ($_POST['name'] ?? ''));
        $email = strtolower(trim((string) ($_POST['email'] ?? '')));
        $password = (string) ($_POST['password'] ?? '');
        $passwordConfirm = (string) ($_POST['password_confirm'] ?? '');

        $errors = [];
        if ($name === '' || mb_strlen($name) > 120) {
            $errors[] = 'الاسم مطلوب وبحد أقصى 120 حرف.';
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'صيغة البريد الإلكتروني غير صحيحة.';
        } elseif ($this->aclModel->emailExists($email)) {
            $errors[] = 'البريد الإلكتروني مستخدم مسبقاً.';
        }

        if (strlen($password) < 8) {
            $errors[] = 'كلمة المرور يجب أن تكون 8 أحرف على الأقل.';
        }

        if ($password !== $passwordConfirm) {
            $errors[] = 'تأكيد كلمة المرور غير متطابق.';
        }

        $defaultRoleId = $this->resolveDefaultRoleId();
        if ($defaultRoleId === null) {
            $errors[] = 'التسجيل غير متاح حالياً. تواصل مع مسؤول النظام.';
        }

        if ($errors !== []) {
            set_old_input([
                'name' => $name,
                'email' => $email,
            ]);
            flash('error', implode(' ', $errors));
            redirect('/register');
        }

        $now = date('Y-m-d H:i:s');
        $userId = UuidService::v4();

        $this->aclModel->createUser([
            'id' => $userId,
            'name' => $name,
            'email' => $email,
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'is_active' => 1,
            'created_at' => $now,
            'updated_at' => $now,
        ]);
        $this->aclModel->setUserRoles($userId, [$defaultRoleId]);

        if (!AuthService::login($email, $password)) {
            flash('success', 'تم إنشاء الحساب بنجاح. يمكنك تسجيل الدخول الآن.');
            redirect('/login');
        }

        flash('success', 'تم إنشاء الحساب بنجاح.');
        redirect('/');
    }

    public function logout(): void
    {
        AuthService::logout();
        flash('success', 'تم تسجيل الخروج.');
        redirect('/login');
    }

    private function resolveDefaultRoleId(): ?string
    {
        $candidates = ['user', 'viewer', 'manager'];
        foreach ($candidates as $roleName) {
            $roleId = $this->aclModel->findRoleIdByName($roleName, true);
            if ($roleId !== null) {
                return $roleId;
            }
        }

        return null;
    }
}
